package resulter

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"log"
	"mime/multipart"
	"net"
	"net/http"
	"strings"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/noc/go/startrek"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/event"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/service"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/util"
)

type General struct {
	StToken       string `yaml:"st_token"`
	StQueue       string `yaml:"st_queue"`
	CronDailyPlan int    `yaml:"cron_daily"`
	HECToken      string `yaml:"hec_token"`
	HECUrl        string `yaml:"hec_url"`
	Debug         bool   `yaml:"debug"`
}

type WebResponse struct {
	Service        string              `json:"service"`
	HostsPerSource map[string][]string `json:"hosts"`
	TimeDelta      int                 `json:"time_delta"`
}

const (
	YadiReportType     = "yadi"
	CoverageReportType = "coverage"
)

func buildAttachBuffer(data map[string][]string) []byte {
	out := ""
	for key := range data {
		out += fmt.Sprintf("%s\n", key)
		for _, value := range data[key] {
			out += fmt.Sprintf("%s\n", value)
		}
	}
	return []byte(out)
}

func submitAttach(issue string, token string, fileContents []byte) {
	url := fmt.Sprintf("https://st-api.yandex-team.ru/v2/issues/%s/attachments?filename=list.txt", issue)
	body := new(bytes.Buffer)
	writer := multipart.NewWriter(body)
	part, err := writer.CreateFormFile("file", "list.txt")
	if err != nil {
		return
	}
	_, e := part.Write(fileContents)
	if e != nil {
		return
	}
	err = writer.Close()
	if err != nil {
		return
	}

	req, err := http.NewRequest("POST", url, body)
	if err != nil {
		log.Fatal(err)
	}
	req.Header.Set("Content-Type", writer.FormDataContentType())
	req.Header.Add("Authorization", fmt.Sprintf("OAuth %s", token))
	client := &http.Client{}
	res, err := client.Do(req)
	if err != nil {
		log.Fatal(err)
	}
	defer res.Body.Close()
	log.Printf("ST submit for %s response code %d\n", issue, res.StatusCode)
}

func SubmitReportToHEC(conf *General, service service.ConfService, diffMap map[string][]string) error {
	entries, _ := BuildSplunkEntries(conf, service, diffMap)
	limit := 100
	for base := 0; base < len(entries); base += limit {
		bound := base + limit
		if bound >= len(entries) {
			bound = len(entries)
		}
		jsonBytes, marshallErr := json.Marshal(entries[base:bound])
		if marshallErr != nil {
			log.Printf("Unable to marshal")
			return marshallErr
		}
		submitErr := util.SubmitToSplunk(conf.HECUrl, conf.HECToken, jsonBytes)
		if submitErr != nil {
			log.Println("Unablle to submit to splunk", submitErr)
		}
	}
	return nil
}

func BuildSplunkEntries(conf *General, service service.ConfService, diffMap map[string][]string) ([]*event.SplunkEntry, error) {
	serviceSlug := service.Slug
	scanID := uuid.Must(uuid.NewV4()).String()
	out := make([]*event.SplunkEntry, 0)
	for dservice := range diffMap {
		for _, host := range diffMap[dservice] {
			out = append(out, event.NewSplunkEntry(scanID, host, serviceSlug, dservice, service.OsqueryTag))
		}
	}
	return out, nil
}

func buildIssue(conf *General, service service.ConfService, data map[string][]string, reportType string) map[string]interface{} {
	var queue = &startrek.Entity{Key: conf.StQueue}
	var ticketMsg string
	var ticketHeader string

	switch reportType {
	case YadiReportType:
		templateMsg := fmt.Sprintf("Hi\nLooks like there are several vulnerable packages found by osquery at %s\n", service.Slug)
		for hostname := range data {
			if len(data[hostname]) != 0 {
				templateMsg += fmt.Sprintf("Vulnerable package %s found on \n", hostname)
				templateMsg += fmt.Sprintf("<{View hosts\n%%%%%s%%%%\n}>", strings.Join(data[hostname], "\n"))
			}
		}
		ticketMsg = templateMsg
		ticketHeader = "[osquery] Vulnerable packages"
	case CoverageReportType:
		templateMsg := fmt.Sprintf("Hi\nLooks like there are multiple problems with osquery at %s\n", service.Slug)
		for source := range data {
			// there are no hosts found
			if len(data[source]) == 0 {
				templateMsg = fmt.Sprintf("Hi\nIt seems, that osquery work on all instances at %s\n", service.Slug)
				// we have found some hosts
			} else {
				templateMsg += fmt.Sprintf("There are no any logs from %d host for at least %d hours\n", len(data[source]),
					service.TimeDelta)
				templateMsg += fmt.Sprintf("<{View hosts\n%%%%\n%s\n%%%%\n}>", strings.Join(data[source], "\n"))
			}
		}
		ticketMsg = templateMsg
		ticketHeader = "[osquery] Coverage issue"
	}
	return map[string]interface{}{
		"queue":       queue,
		"summary":     ticketHeader,
		"description": ticketMsg,
	}
}

func CreateSTReport(conf *General, service service.ConfService, diffMap map[string][]string, reportType string) error {
	var token = conf.StToken
	st := startrek.Client{
		APIV2URL:  "https://st-api.yandex-team.ru/v2",
		APIToken:  token,
		UserAgent: "Go Get Issue example",
		Client: http.Client{
			Transport: &http.Transport{
				Proxy: http.ProxyFromEnvironment,
				Dial: (&net.Dialer{
					Timeout:   10 * time.Second,
					KeepAlive: 30 * time.Second,
				}).Dial,
				TLSHandshakeTimeout: 10 * time.Second,
			},
			Timeout: 30 * time.Second,
		},
	}
	issue := buildIssue(conf, service, diffMap, reportType)
	attachBuf := buildAttachBuffer(diffMap)
	var respQueue startrek.Issue
	errTicket := st.CreateIssue(context.Background(), issue, &respQueue)
	if errTicket != nil {
		log.Printf("Error creating ticket\n")
		log.Println(errTicket)
		return errTicket
	}
	submitAttach(respQueue.Key, conf.StToken, attachBuf)
	log.Printf("Created issues id = %s", respQueue.Key)
	return nil
}

func CreateHandlerResponseJSON(conf *General, service service.ConfService, diffMap map[string][]string, reportType string) *WebResponse {
	return &WebResponse{
		service.Slug,
		diffMap,
		service.TimeDelta,
	}
}
