package service

import (
	"fmt"
	"io/ioutil"
	"log"
	"regexp"

	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/compute"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/conductor"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/sourceclient"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/splunk"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/util"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/yadi"
)

const (
	ConductorName = conductor.Name
	SplunkName    = splunk.Name
	ComputeName   = compute.Name
	YadiCSVFile   = "/tmp/yadi_test.csv"
)

type Service interface {
	GetMissingHosts() (map[string][]string, error)
	GetVulnerablePackages() (map[string][]string, error)
	GetSource() []string
	EnableDebug()
}

type (
	ConfService struct {
		Slug              string   `yaml:"slug"`
		Sources           []string `yaml:"sources"`
		OsqueryTag        string   `yaml:"osquery_tag"`
		TimeDelta         int      `yaml:"time_delta"`
		ConductorGroup    string   `yaml:"conductor_group"`
		SkipList          []string `yaml:"skip_list"`
		YadiCheck         bool     `yaml:"yadi_check"`
		YadiCheckSeverity string   `yaml:"yadi_severity"`
		YadiCheckTag      string   `yaml:"yadi_osquery_tag"`
		ConductorConf     *conductor.Source
		SplunkConf        *splunk.Source
		ComputeConf       *compute.Source
		DebugEnabled      bool
	}
)

// GetMissingHosts returns diffs against conductor results
func (s *ConfService) GetMissingHosts() (map[string][]string, error) {
	lookupMap := make(map[string]bool)
	diffMap := make(map[string][]string)
	if s.Slug == "ycloud" {
		s.ComputeConf.ComputeJWT = util.GetIAMToken("https://iam.api.cloud.yandex.net/iam/v1/tokens", s.ComputeConf.ComputeProdSA, s.ComputeConf.ComputeProdKeyID, s.ComputeConf.ProdPrivKeyPath)
		s.ComputeConf.ComputePreprodJWT = util.GetIAMToken("https://iam.api.cloud-preprod.yandex.net/iam/v1/tokens", s.ComputeConf.ComputePreprodSA, s.ComputeConf.ComputePreprodKeyID, s.ComputeConf.PreprodPrivKeyPath)
		diffMap[ComputeName], _ = GetHostsList(s, ComputeName)
		return diffMap, nil
	}
	conductorHosts, fetchErr := GetHostsList(s, ConductorName)
	for _, host := range conductorHosts {
		lookupMap[host] = true
	}
	if fetchErr != nil {
		return nil, fetchErr
	}
	for _, source := range s.Sources {
		if source != ConductorName {
			hosts, ferror := GetHostsList(s, source)
			if ferror == nil {
				diffList := make([]string, 0)
				checkMap := make(map[string]bool)
				for _, host := range hosts {
					checkMap[host] = true
				}
				for _, host := range conductorHosts {
					if _, ok := checkMap[host]; !ok && host != "" {
						diffList = append(diffList, host)
					}
				}
				diffMap[source] = diffList
			} else {
				log.Printf("Fetch error\n")
				log.Println(ferror)
			}
		}
	}
	return diffMap, nil
}

func (s *ConfService) GetSource() []string {
	return s.Sources
}

func (s *ConfService) EnableDebug() {
	s.DebugEnabled = true
}

func (s *ConfService) GetVulnerablePackages() (map[string][]string, error) {
	var csvBytes []byte
	var checker yadi.Checker
	checkConf := yadi.NewPkgCheckerConf(s.YadiCheckSeverity, s.YadiCheckTag)
	if !s.DebugEnabled {
		hosts, _ := GetHostsList(s, ConductorName)
		splunkConfg := s.SplunkConf
		log.Printf("%d hosts of %s found\n", len(hosts), s.Slug)
		splunkQuery := checker.GetSplunkQuery(checkConf)
		csvBytes, _ = splunkConfg.RequestToCsv(splunkQuery)
	} else {
		csvBytes, _ = ioutil.ReadFile(YadiCSVFile)
	}
	reader, _ := checker.GetPkgScanReader(csvBytes)
	vulnPkgPerHost := checker.GetCheckResult(checkConf, reader)
	return vulnPkgPerHost, nil
}

func GetHostsList(serviceConf *ConfService, sourceType string) ([]string, error) {
	//skipMap := make(map[string]bool)
	regexList := make([]*regexp.Regexp, 0)
	hosts := make([]string, 0)
	for _, regexStr := range serviceConf.SkipList {
		//skipMap[hostname] = true
		r, _ := regexp.Compile(regexStr)
		regexList = append(regexList, r)
	}
	fetchClient := NewFetchClient(serviceConf, sourceType)
	out, err := fetchClient.FetchHosts()
	for _, hostname := range out {
		for _, regexToSkip := range regexList {
			if !regexToSkip.MatchString(hostname) {
				hosts = append(hosts, hostname)
			} else {
				fmt.Print(hostname)
			}
		}
	}
	return hosts, err
}

func NewFetchClient(serviceConf *ConfService, sourceType string) sourceclient.SourceClient {
	switch sourceType {
	case ConductorName:
		return &conductor.Source{
			ConductorURL: fmt.Sprintf("%s%s", serviceConf.ConductorConf.ConductorURL, serviceConf.ConductorGroup),
		}
	case SplunkName:
		updatedConf := serviceConf.SplunkConf
		updatedConf.TimeDelta = serviceConf.TimeDelta
		updatedConf.Tag = serviceConf.OsqueryTag
		return updatedConf
	case ComputeName:
		computeIamTokenRoute := fmt.Sprintf("%s%s", serviceConf.ComputeConf.ComputeIamHost, serviceConf.ComputeConf.ComputeIamTokenPath)
		iamToken, _ := compute.GetIAMToken(computeIamTokenRoute, serviceConf.ComputeConf.OauthToken)
		return &compute.Source{
			OauthToken:        serviceConf.ComputeConf.OauthToken,
			IamToken:          iamToken,
			ComputeIamHost:    computeIamTokenRoute,
			TimeDelta:         serviceConf.TimeDelta,
			SplunkUser:        serviceConf.ComputeConf.SplunkUser,
			SplunkPwd:         serviceConf.ComputeConf.SplunkPwd,
			SplunkAuthPath:    serviceConf.ComputeConf.SplunkAuthPath,
			ComputeJWT:        serviceConf.ComputeConf.ComputeJWT,
			ProdFolders:       serviceConf.ComputeConf.ProdFolders,
			PreprodFolders:    serviceConf.ComputeConf.PreprodFolders,
			ComputePreprodJWT: serviceConf.ComputeConf.ComputePreprodJWT,
		}
	default:
		return nil
	}
}
