package splunk

import (
	"bytes"
	"crypto/tls"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"strings"
	"time"

	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/util"
)

const (
	Name = "splunk"
)

type Source struct {
	Host          string `yaml:"host"`
	JobsPath      string `yaml:"jobs_path"`
	AuthPath      string `yaml:"auth_path"`
	SearchFmt     string `yaml:"search_fmt"`
	Tag           string `yaml:"tag"`
	TimeDelta     int
	UsernameCreds string `yaml:"username"`
	PasswordCreds string `yaml:"password"`
	SessionKey    string
}

func (c *Source) makeSplunkRequest() ([]byte, error) {
	data := url.Values{}
	searchStr := fmt.Sprintf(c.SearchFmt, c.TimeDelta, c.Tag)
	data.Set("search", searchStr)
	data.Set("output_mode", "csv")
	tr := &http.Transport{
		TLSClientConfig: &tls.Config{InsecureSkipVerify: true},
	}
	client := &http.Client{Transport: tr, Timeout: 360 * time.Second}

	req, _ := http.NewRequest("POST",
		fmt.Sprintf("%s%s", c.Host, c.JobsPath), bytes.NewBufferString(data.Encode()))
	req.Header.Add("Authorization", fmt.Sprintf("Splunk %s", c.SessionKey))
	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")
	resp, err := client.Do(req)
	log.Printf("Processing Splunk request\n")
	if err != nil {
		return nil, err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
			log.Fatal(err)
		}
	}()
	log.Printf("Reading Splunk response body\n")
	rbody, readErr := ioutil.ReadAll(resp.Body)
	return rbody, readErr
}

func (c *Source) FetchHosts() ([]string, error) {
	err := c.UpdateSessionKey()
	if err != nil {
		log.Fatal("Unable to retrieve Splunk session key, exiting...")
	}
	out := make([]string, 0)
	body, readErr := c.makeSplunkRequest()
	if readErr != nil {
		log.Printf("Unable to read response")
		return out, readErr
	}
	if len(body) == 0 {
		return nil, errors.New("splunk: unable to parse response")
	}
	sbody := string(body)
	hosts := strings.Split(sbody, "\n")
	for _, h := range hosts[1:] {
		if len(h) > 1 {
			trimmed := strings.Replace(h, "\"", "", -1)
			spl := strings.Split(trimmed, ",")
			if len(spl) > 1 {
				out = append(out, spl[0])
			} else {
				return out, errors.New("splunk: unable to parse response")
			}
		}
	}
	return out, nil
}

func (c *Source) GetSourceType() string {
	_, err := util.GetSplunkSession(c.UsernameCreds, c.PasswordCreds, c.Host, c.AuthPath)
	if err != nil {
		log.Printf("Unable to retrieve Splunk session key")
		log.Println(err)
	}
	return Name
}

func (c *Source) UpdateSessionKey() error {
	//key, err := c.getSessionKey()
	key, err := util.GetSplunkSession(c.UsernameCreds, c.PasswordCreds, c.Host, c.AuthPath)
	if err != nil {
		return err
	}
	c.SessionKey = key
	return nil
}

func (c *Source) RequestToCsv(searchQuery string) ([]byte, error) {
	err := c.UpdateSessionKey()
	if err != nil {
		log.Fatal("Unable to retrieve Splunk session key, exiting...")
	}
	log.Printf("fetching splunk\n")
	data := url.Values{}
	//searchStr := fmt.Sprintf(c.SearchFmt, c.TimeDelta, c.Tag)
	data.Set("search", searchQuery)
	data.Set("output_mode", "csv")
	tr := &http.Transport{
		TLSClientConfig: &tls.Config{InsecureSkipVerify: true},
	}
	client := &http.Client{Transport: tr, Timeout: 360 * time.Second}

	req, _ := http.NewRequest("POST",
		fmt.Sprintf("%s%s", c.Host, c.JobsPath), bytes.NewBufferString(data.Encode()))
	req.Header.Add("Authorization", fmt.Sprintf("Splunk %s", c.SessionKey))
	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")
	resp, err := client.Do(req)
	log.Printf("Processing Splunk request\n")
	if err != nil {
		return nil, err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
			log.Fatal(err)
		}
	}()
	log.Printf("Reading Splunk response body\n")
	body, readErr := ioutil.ReadAll(resp.Body)
	if readErr != nil {
		log.Printf("Unable to read response")
	}
	return body, readErr
}
