package util

import (
	"bytes"
	"crypto/rsa"
	"crypto/tls"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"regexp"
	"strings"
	"time"

	"github.com/golang-jwt/jwt/v4"
)

func GetEpochTime() int64 {
	t := time.Now()
	return t.Unix()
}

func SubmitToSplunk(submitURL string, splunkToken string, data []byte) error {
	tr := &http.Transport{
		TLSClientConfig: &tls.Config{InsecureSkipVerify: true},
	}
	req, _ := http.NewRequest("POST", submitURL, bytes.NewReader(data))
	req.Header.Set("Authorization", fmt.Sprintf("Splunk %s", splunkToken))
	req.Header.Set("Content-Type", "application/json")
	cl := &http.Client{Transport: tr}
	resp, err := cl.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	log.Printf("response Status: %s %d\n", resp.Status, resp.StatusCode)
	if resp.StatusCode != 200 {
		return fmt.Errorf("HEC response code = %d", resp.StatusCode)
	}
	return nil
}

func signedToken(apiURL string, serviceAccountID string, keyID string, keyFile string) string {
	issuedAt := time.Now()
	token := jwt.NewWithClaims(ps256WithSaltLengthEqualsHash, jwt.StandardClaims{
		Issuer:    serviceAccountID,
		IssuedAt:  issuedAt.Unix(),
		ExpiresAt: issuedAt.Add(time.Hour).Unix(),
		Audience:  apiURL,
	})
	token.Header["kid"] = keyID

	privateKey := loadPrivateKey(keyFile)
	signed, err := token.SignedString(privateKey)
	if err != nil {
		panic(err)
	}
	return signed
}

var ps256WithSaltLengthEqualsHash = &jwt.SigningMethodRSAPSS{
	SigningMethodRSA: jwt.SigningMethodPS256.SigningMethodRSA,
	Options: &rsa.PSSOptions{
		SaltLength: rsa.PSSSaltLengthEqualsHash,
	},
}

func loadPrivateKey(keyFile string) *rsa.PrivateKey {
	//TODO: remove it
	var data []byte
	if keyFile[0] == '/' {
		data, _ = ioutil.ReadFile(keyFile)
	} else {
		data, _ = base64.StdEncoding.DecodeString(keyFile)
	}
	rsaPrivateKey, err := jwt.ParseRSAPrivateKeyFromPEM(data)
	if err != nil {
		panic(err)
	}
	return rsaPrivateKey
}

func GetIAMToken(computeURL string, serviceAccountID string, keyID string, keyFile string) string {
	jot := signedToken(computeURL, serviceAccountID, keyID, keyFile)
	resp, err := http.Post(
		computeURL,
		"application/json",
		strings.NewReader(fmt.Sprintf(`{"jwt":"%s"}`, jot)),
	)
	if err != nil {
		panic(err)
	}
	defer resp.Body.Close()
	if resp.StatusCode != http.StatusOK {
		body, _ := ioutil.ReadAll(resp.Body)
		panic(fmt.Sprintf("%s: %s", resp.Status, body))
	}
	var data struct {
		IAMToken string `json:"iamToken"`
	}
	err = json.NewDecoder(resp.Body).Decode(&data)
	if err != nil {
		panic(err)
	}
	return data.IAMToken
}

func GetSplunkSession(username, pwd, hostname, authEndpoint string) (string, error) {
	log.Printf("retrieving session key\n")
	regexPatthern := "\\<sessionKey\\>(.*?)\\<"
	formData := url.Values{
		"username": {username},
		"password": {pwd},
	}
	http.DefaultTransport.(*http.Transport).TLSClientConfig = &tls.Config{InsecureSkipVerify: true}
	resp, submitErr := http.PostForm(fmt.Sprintf("%s%s", hostname, authEndpoint), formData)
	if submitErr != nil {
		return "", submitErr
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
			log.Fatal(err)
		}
	}()
	body, _ := ioutil.ReadAll(resp.Body)
	sbody := string(body)
	re := regexp.MustCompile(regexPatthern)
	matches := re.FindStringSubmatch(sbody)
	if len(matches) < 2 {
		return "", nil
	}
	return matches[1], nil
}
