package workers

import (
	"log"
	"sync"

	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/config"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/resulter"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/service"
)

func Run(cfg config.Config) {
	services := cfg.Services
	var wg sync.WaitGroup
	for _, s := range services {
		s.ConductorConf = &cfg.ConductorConf
		s.SplunkConf = &cfg.SplunkConf
		s.ComputeConf = &cfg.ComputeConf
		if cfg.GeneralConf.Debug {
			s.EnableDebug()
		}
		if s.YadiCheck {
			wg.Add(1)
			go Yadi(cfg, s, &wg)
		}
		wg.Add(1)
		go Coverage(cfg, s, &wg)
	}
	wg.Wait()
}

func Yadi(cfg config.Config, service service.ConfService, wg *sync.WaitGroup) {
	defer wg.Done()
	if service.YadiCheckTag == "" {
		service.YadiCheckTag = service.OsqueryTag
	}
	packages, cerr := service.GetVulnerablePackages()
	if cerr != nil {
		log.Printf("Unable to check packages for %s\n", service.Slug)
	}
	if service.DebugEnabled {
		for hostname, pkgs := range packages {
			log.Printf("Vulnerable packages for %s\n", hostname)
			for _, pkg := range pkgs {
				log.Printf("%s \n", pkg)
			}
		}
	} else {
		//hecErr := resulter.SubmitReportToHEC(&cfg.GeneralConf, service, diffMap)
		reportErr := resulter.CreateSTReport(&cfg.GeneralConf, service, packages, resulter.YadiReportType)
		if reportErr != nil {
			log.Printf("Report error")
		}
	}
}

func Coverage(cfg config.Config, service service.ConfService, wg *sync.WaitGroup) {
	defer wg.Done()
	diffMap, err := service.GetMissingHosts()
	if err != nil {
		log.Printf("Unable to find missing hosts for %s service", service.Slug)
	}
	hecErr := resulter.SubmitReportToHEC(&cfg.GeneralConf, service, diffMap)
	reportErr := resulter.CreateSTReport(&cfg.GeneralConf, service, diffMap, resulter.CoverageReportType)
	if reportErr != nil || hecErr != nil {
		log.Printf("Report error")
	}
}
