package awscommon

import (
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	awsClient "github.com/aws/aws-sdk-go/aws/client"
	awsCredentials "github.com/aws/aws-sdk-go/aws/credentials"
	awsSession "github.com/aws/aws-sdk-go/aws/session"
	"golang.org/x/net/http2"
)

const (
	// Set maximum idle connections to something large as we can have many concurrent requests in s3Merger.
	maxIdleConns    = 1000
	idleConnTimeout = time.Second * 30

	// Send HTTP/2 pings each readIdleTimeout if connection is idle.
	readIdleTimeout = time.Second * 30

	retryBackoff = time.Second * 5
)

type SessionParams struct {
	Endpoint            string
	Region              string
	AccessKeyID         string
	SecretAccessKeyFile string
	NumRetries          int
	RequestTimeout      time.Duration
	EnableVerboseDebug  bool
}

func NewSession(params *SessionParams) (*awsSession.Session, error) {
	secretAccessKeyBytes, err := ioutil.ReadFile(params.SecretAccessKeyFile)
	if err != nil {
		return nil, err
	}
	secretAccessKey := strings.TrimSpace(string(secretAccessKeyBytes))
	credentials := awsCredentials.NewStaticCredentials(params.AccessKeyID, secretAccessKey, "")

	awsLogger := aws.LoggerFunc(func(i ...interface{}) {
		if isReqErrorLog(i) {
			log.Println("AWS SDK ERROR:", fmt.Sprint(i...))
		} else if params.EnableVerboseDebug {
			log.Println("AWS SDK:", fmt.Sprint(i...))
		}
	})

	httpTransport := http.DefaultTransport.(*http.Transport).Clone()
	httpTransport.MaxIdleConns = maxIdleConns
	httpTransport.MaxIdleConnsPerHost = maxIdleConns
	httpTransport.IdleConnTimeout = idleConnTimeout
	h2Transport, err := http2.ConfigureTransports(httpTransport)
	if err != nil {
		return nil, err
	}
	h2Transport.ReadIdleTimeout = readIdleTimeout

	return awsSession.NewSessionWithOptions(awsSession.Options{
		Config: aws.Config{
			Credentials: credentials,
			Endpoint:    &params.Endpoint,
			Region:      &params.Region,
			Retryer: awsClient.DefaultRetryer{
				NumMaxRetries: params.NumRetries,
				MinRetryDelay: retryBackoff,
			},
			HTTPClient: &http.Client{
				Transport: httpTransport,
				Timeout:   params.RequestTimeout,
			},
			Logger:   awsLogger,
			LogLevel: aws.LogLevel(aws.LogDebugWithRequestErrors | aws.LogDebugWithRequestRetries),
		},
	})
}

func isReqErrorLog(i []interface{}) bool {
	if len(i) == 1 {
		if s, ok := i[0].(string); ok {
			// See debugLogReqError() in github.com/aws/aws-sdk-go/request/request.go
			if strings.Contains(s, " failed, ") && strings.Contains(s, ", error ") {
				return true
			}
			// See prepareRetry() in github.com/aws/aws-sdk-go/request/request.go
			if strings.HasPrefix(s, "DEBUG: Retrying Request") {
				return true
			}
		}
	}
	return false
}
