package config

import (
	"encoding/base64"
	"fmt"
	"io/ioutil"
	"net"
	"os"

	"github.com/c2h5oh/datasize"
	"gopkg.in/yaml.v2"
)

const (
	defaultSplunkQueueLength = 1000
	defautlSplunkWorkers     = 100
)

type HostConfig struct {
	EnrollSecret   string `yaml:"enroll_secret"`
	InsecureEnroll bool   `yaml:"insecure_enroll,omitempty"`

	SplunkURL       string `yaml:"url"`
	SplunkToken     string `yaml:"token"`
	SplunkTokenFile string `yaml:"token_file"`
}

type SplunkConfig struct {
	QueueLength int `yaml:"queue_length"`
	Workers     int `yaml:"workers"`
}

type ClickhouseConfig struct {
	// List of Clickhouse hosts.
	Hosts []string `yaml:"hosts"`
	// Clickhouse port, defaults to 9440
	Port int
	// Connection parameters (e.g. {database: yc-osquery, username: yc-osquery, secure: true})
	ConnectionParams map[string]string `yaml:"connection_params"`
	// Path to file containing database password, if not empty, will be appended to connection paramaters.
	PasswordFile string `yaml:"password_file"`

	// Event names for events which should be stored in Clickhouse. Accepts regular expressions.
	EventNames []string `yaml:"event_names"`
	// Event names for events which should be stored ONLY in Clickhouse (will not be stored in Splunk).
	// Accepts regular expressions. Will be automatically included in event_names.
	ClickhouseOnlyEventNames []string `yaml:"clickhouse_only_event_names"`
	// Remove prefixes from event names.
	RemovePrefix []string `yaml:"remove_prefix"`
	// Remove suffixes from event names.
	RemoveSuffix []string `yaml:"remove_suffix"`

	// If true, two tables will be created for each event type: one will be named "shard_{event name}" and store the actual data
	// and one will be named "{event name}" and be a distributed table over shards (see https://clickhouse.com/docs/en/engines/table-engines/special/distributed/)
	EnableSharding bool `yaml:"enable_sharding"`

	// Drop data older than the given amount of days. Defaults to 30 days.
	DropAfterDays int `yaml:"drop_after_days"`
	// Set custom data retention for specific tables.
	CustomDropAfterDays map[string]int `yaml:"custom_drop_after_days"`

	// Maximum time to delay the events in memory before inserting into Clickhouse.
	MaxDelay string `yaml:"max_delay"`
	// Maximum memory to use for batching the results before inserting into Clickhouse.
	MaxMemory datasize.ByteSize `yaml:"max_memory"`
	// Maximum number of connections to Clickhouse.
	MaxConnections int `yaml:"max_connections"`

	// Number of retries to insert, defaults to 3.
	NumRetries int `yaml:"num_retries"`

	EnableDebug bool `yaml:"debug,omitempty"`
}

type S3Config struct {
	AccessKeyID string `yaml:"access_key_id"`
	// Path to file containing secret access key matching access_key_id.
	SecretAccessKeyFile string `yaml:"secret_access_key_file"`

	// S3 endpoint, e.g. storage.yandexcloud.net for Yandex.Cloud PROD or storage.cloud-preprod.yandex.net for Yandex.Cloud PREPROD
	Endpoint string `yaml:"endpoint"`
	Region   string `yaml:"region"`
	Bucket   string `yaml:"bucket"`

	// Which compression algorithm to use. Currently supported: lz4, snappy and none, defaults to lz4.
	Compression string `yaml:"compression"`

	// Merges objects into daily object, defaults to true
	MergeDaily *bool `yaml:"merge_daily"`
	// Timezone will be used for determining when the day starts. If empty, uses the system timezone.
	Timezone string `yaml:"timezone"`
	// Deletes merged objects older than the given number of days. Defaults to 730 days. Note: this option effectively
	// works only when merge_daily is set to true. Normal objects are not deleted
	DeleteMergedAfterDays *int `yaml:"delete_merged_after_days"`

	// Remove prefixes from event names.
	RemovePrefix []string `yaml:"remove_prefix"`
	// Remove suffixes from event names.
	RemoveSuffix []string `yaml:"remove_suffix"`

	// Maximum time to delay the events in memory before storing to S3.
	MaxDelay string `yaml:"max_delay"`
	// Maximum memory to use for batching the results before storing to S3.
	MaxMemory datasize.ByteSize `yaml:"max_memory"`
	// Maximum number of workers storing to S3.
	MaxWorkers int `yaml:"max_workers"`

	// Number of retries to upload object, defaults to 5.
	NumRetries int `yaml:"num_retries"`

	EnableDebug bool `yaml:"debug,omitempty"`

	EnableVerboseDebug bool `yaml:"verbose_debug,omitempty"`
}

type KinesisConfig struct {
	AccessKeyID string `yaml:"access_key_id"`
	// Path to file containing secret access key matching access_key_id.
	SecretAccessKeyFile string `yaml:"secret_access_key_file"`

	// YDS endpoint, e.g. yds.serverless.yandexcloud.net/ru-central1/abc/def/
	Endpoint string `yaml:"endpoint"`
	Region   string `yaml:"region"`
	Stream   string `yaml:"stream"`
	// Limit the number of events to send in one request.
	MaxLogsPerBatch int `yaml:"max_logs_per_batch"`

	// Kinesis records are done via separate pool of workers. The queue length is bounded.
	QueueLength int `yaml:"queue_length"`
	// Maximum number of workers pushing records to Kinesis. Equals number of CPUs by default.
	MaxWorkers int `yaml:"max_workers"`
	// Slow the worker on failure for the given amount of milliseconds.
	BackoffOnFailMillis int `yaml:"backoff_on_fail_ms"`

	// Percent of logs to send to Kinesis (0-100).
	Percent *int `yaml:"percent"`
	// Percent of logs to send to Kinesis for specific event name (0-100).
	PercentPerEventName map[string]int `yaml:"percent_per_event_name"`
	// Hostnames (* pattern allowed) for which all logs will be sent to Kinesis without sampling.
	SendAllForHosts []string `yaml:"send_all_for_hosts"`

	EnableDebug        bool `yaml:"debug,omitempty"`
	EnableVerboseDebug bool `yaml:"verbose_debug,omitempty"`
}

type SyslogConfig struct {
	Servers []SyslogServerConfig `yaml:"servers"`
}

type SyslogServerConfig struct {
	Name         string `yaml:"name"` // Will be set in ParsedEvent.Name
	Address      string `yaml:"address"`
	Parser       string `yaml:"parser"`
	EventHandler string `yaml:"event_handler"` // By default, sends ParsedEvent to SendMgr
	UseTCP       bool   `yaml:"use_tcp"`       // By default, UDP will be used
}

// Subnet type is used to add custom yaml marshalling.
type Subnet struct {
	*net.IPNet
}

type AddForPeersConfig struct {
	// Add the following values for peers from these subnets. Peers are matched both by X-Forwarded-For and network peer
	// address.
	Subnets []Subnet `yaml:"subnets"`
	// Values is the list of key-value pairs to be added to each message from matching peers.
	Values map[string]string `yaml:"values"`
}

type SenderConfig struct {
	// Server configuration.
	EnableTLS                     bool     `yaml:"tls,omitempty"`
	Port                          int      `yaml:"port,omitempty"`
	HealthcheckPort               int      `yaml:"healthcheck_port,omitempty"`
	ManagementPort                int      `yaml:"management_port,omitempty"`
	TLSKeyFile                    string   `yaml:"key,omitempty"`
	TLSCertificateFile            string   `yaml:"cert,omitempty"`
	AdditionalTLSKeyFiles         []string `yaml:"additional_keys,omitempty"`
	AdditionalTLSCertificateFiles []string `yaml:"additional_certs,omitempty"`
	MaxConcurrentRequests         int      `yaml:"max_concurrent_requests"`
	MaxConcurrentConnections      int      `yaml:"max_concurrent_connections"`

	EnableDebug bool `yaml:"debug,omitempty"`

	// Map keys are destination hostnames (the domains which osquery-sender listens to).
	HostsConfig map[string]*HostConfig `yaml:"hosts"`

	EnrollmentHmacSecret string `yaml:"enrollment_hmac_secret"`

	// Splunk writer configuration (part of the splunk configuration is in the HostsConfig).
	Splunk      *SplunkConfig `yaml:"splunk"`
	TLSInsecure bool          `yaml:"insecure"`

	// Osquery log parser config.
	// Clickhouse and S3 writers will add data from the decorators to the list of columns (Splunk and Kinesis pass
	// through all data).
	AddDecorators []string `yaml:"add_decorators"`
	// The fieldDefaults to hostIdentifier
	HostnameKeys []string `yaml:"hostname_keys"`
	// Add fields to osquery logs depending on the source.
	AddForPeers []AddForPeersConfig `yaml:"add_for_peers"`

	// Clickhouse writer configuration.
	Clickhouse *ClickhouseConfig `yaml:"clickhouse"`

	// S3 writer configuration.
	S3 *S3Config `yaml:"s3"`

	// Kinesis writer configuration.
	Kinesis *KinesisConfig `yaml:"kinesis"`

	// Syslog server + parser configuration.
	Syslog *SyslogConfig `yaml:"syslog"`
}

func FromFile(filename string) (*SenderConfig, error) {
	result := new(SenderConfig)
	result.EnableTLS = false
	if os.Getenv("ENV") == "DEPLOY" {
		data := os.Getenv("CONFIG_DATA")
		decoded, _ := base64.RawStdEncoding.DecodeString(data)
		if err := yaml.Unmarshal(decoded, &result); err != nil {
			return nil, err
		}
	} else {
		if _, err := os.Stat(filename); os.IsNotExist(err) {
			return nil, err
		}
		data, err := ioutil.ReadFile(filename)
		if err != nil {
			return nil, err
		}
		if err := yaml.Unmarshal(data, &result); err != nil {
			return nil, err
		}
	}
	if result.Splunk == nil {
		result.Splunk = &SplunkConfig{
			QueueLength: defaultSplunkQueueLength,
			Workers:     defautlSplunkWorkers,
		}
	}
	if result.HostnameKeys == nil {
		result.HostnameKeys = []string{"hostIdentifier"}
	}

	return result, nil
}

func (s *Subnet) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var subnetStr string
	if err := unmarshal(&subnetStr); err != nil {
		return nil
	}

	_, subnet, err := net.ParseCIDR(subnetStr)
	if err != nil {
		return fmt.Errorf("can't parse subnet %s: %v", s, err)
	}
	s.IPNet = subnet
	return nil
}

// MarshalYAML must be called on value receiver.
func (s Subnet) MarshalYAML() (interface{}, error) {
	return s.IPNet.String(), nil
}
