package handlers

import (
	"crypto/hmac"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/osquery/osquery-sender/config"
	"a.yandex-team.ru/security/osquery/osquery-sender/enroll"
	"a.yandex-team.ru/security/osquery/osquery-sender/metrics"
	"a.yandex-team.ru/security/osquery/osquery-sender/sendmgr"
)

func Enroll(ctx echo.Context, config *config.SenderConfig, mgr *sendmgr.SendMgr) error {
	metrics.IncomingRps.Inc()

	dstHostname := ctx.Request().Host
	hostCfg, ok := config.HostsConfig[dstHostname]
	if !ok {
		return ctx.String(http.StatusBadRequest, "configuration for the host "+dstHostname+" not found")
	}

	req := new(enroll.Request)
	if err := ctx.Bind(req); err != nil {
		return ctx.String(http.StatusBadRequest, "bad request format")
	}

	if !hostCfg.InsecureEnroll {
		enrollSecret := []byte(config.HostsConfig[dstHostname].EnrollSecret)
		if !hmac.Equal(enrollSecret, []byte(req.EnrollSecret)) {
			return ctx.JSON(http.StatusOK, &enroll.Response{
				NodeKey:     "",
				NodeInvalid: true,
			})
		}
		nodeKey := enroll.GenerateNodeKey(config.EnrollmentHmacSecret, string(enrollSecret), req.HostIdentifier)
		mgr.LogEnroll(dstHostname, nodeKey, req)

		return ctx.JSON(http.StatusOK, &enroll.Response{
			NodeKey:     nodeKey,
			NodeInvalid: false,
		})
	} else {
		mgr.LogEnroll(dstHostname, enroll.FakeEnrollNodeKeySecret, req)
		return ctx.JSON(http.StatusOK, &enroll.Response{
			NodeKey:     enroll.FakeEnrollNodeKeySecret,
			NodeInvalid: false,
		})
	}
}
