package handlers

import (
	"context"
	"encoding/json"
	"io"
	"log"
	"net"
	"net/http"
	"runtime/trace"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/osquery/osquery-sender/config"
	"a.yandex-team.ru/security/osquery/osquery-sender/enroll"
	"a.yandex-team.ru/security/osquery/osquery-sender/logger"
	"a.yandex-team.ru/security/osquery/osquery-sender/metrics"
	"a.yandex-team.ru/security/osquery/osquery-sender/parser"
	"a.yandex-team.ru/security/osquery/osquery-sender/sendmgr"
)

const (
	veryLongLogMillis = 2500
)

// Log is the logger handler
func Log(ctx echo.Context, conf *config.SenderConfig, mgr *sendmgr.SendMgr) error {
	metrics.IncomingRps.Inc()
	startTime := time.Now()
	traceCtx, traceTask := trace.NewTask(context.Background(), "Log")
	defer traceTask.End()

	hostname := ctx.Request().Host
	peerIP := net.ParseIP(ctx.RealIP())
	trace.Logf(traceCtx, "", "Host %s, peer ip %s", hostname, peerIP)

	hostCfg, ok := conf.HostsConfig[hostname]
	if !ok {
		return ctx.String(http.StatusBadRequest, "configuration for the hostname not found")
	}

	var bodyBytes []byte
	var err error
	if ctx.Request().Body != nil {
		trace.WithRegion(traceCtx, "io.ReadAll", func() {
			bodyBytes, err = io.ReadAll(ctx.Request().Body)
		})
	}
	if err != nil {
		log.Printf("ERROR: Can't read body from %s: %v\n", peerIP, err)
		trace.Logf(traceCtx, "", "Could not read body: %v", err)
		return ctx.String(http.StatusBadRequest, "could not read body")
	}
	metrics.RequestSizeMillis.Add(int64(len(bodyBytes)))
	trace.Logf(traceCtx, "", "Message size: %dkb", len(bodyBytes)/1024)

	// TODO(e-sidorov): check node key
	if bodyBytes != nil {
		var message parser.LogMessage
		trace.WithRegion(traceCtx, "json.Unmarshal", func() {
			err = json.Unmarshal(bodyBytes, &message)
		})
		if err != nil {
			log.Printf("WARNING: Can't parse incoming message: %s...: %v\n", string(bodyBytes[:100]), err)
			trace.Logf(traceCtx, "", "Can't parse: %v", err)
			metrics.IncomingParsingErrors.Inc()
			return ctx.JSON(http.StatusBadRequest, "json format error")
		}

		if !hostCfg.InsecureEnroll {
			if !enroll.CheckNodeKey(conf.EnrollmentHmacSecret, conf.HostsConfig[hostname].EnrollSecret, message.NodeKey) {
				return ctx.JSON(http.StatusOK, &logger.Response{NodeInvalid: true})
			}
		}

		var events []*parser.ParsedEvent
		trace.WithRegion(traceCtx, "parser.ParseIntoEvents", func() {
			events = parser.ParseIntoEvents(&message, peerIP, conf)
		})

		trace.WithRegion(traceCtx, "sendmgr.Log", func() {
			mgr.Log(hostname, events)
		})
	}

	// Do not record times for non-200 return values.
	millis := time.Since(startTime).Milliseconds()
	metrics.RequestTimeMillis.Add(millis)
	if millis >= veryLongLogMillis {
		trace.Logf(traceCtx, "", "Very long Log time: %dms", millis)
	} else {
		trace.Logf(traceCtx, "", "Log time: %dms", millis)
	}

	return ctx.JSON(http.StatusOK, &logger.Response{NodeInvalid: false})
}
