package handlers

import (
	"bytes"
	"io"
	"net/http"
	"runtime/pprof"
	"runtime/trace"
	"strconv"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/osquery/osquery-sender/util"
)

func Profile(context echo.Context) error {
	seconds, err := strconv.Atoi(context.QueryParam("seconds"))
	if err != nil {
		return context.String(http.StatusBadRequest, "seconds must be integer")
	}
	if seconds == 0 {
		seconds = 10
	}

	reader, writer := io.Pipe()
	err = pprof.StartCPUProfile(writer)
	if err != nil {
		return context.String(http.StatusInternalServerError, err.Error())
	}
	go util.RunWithLabels(pprof.Labels("name", "pprof-cpu-profile"), func() {
		time.Sleep(time.Duration(seconds) * time.Second)
		pprof.StopCPUProfile()
		_ = writer.Close()
	})

	return context.Stream(http.StatusOK, "application/octet-stream", reader)
}

func Heap(context echo.Context) error {
	buf := bytes.Buffer{}
	err := pprof.WriteHeapProfile(&buf)
	if err != nil {
		return context.String(http.StatusInternalServerError, err.Error())
	}

	return context.Stream(http.StatusOK, "application/octet-stream", &buf)
}

func Gorotune(context echo.Context) error {
	buf := bytes.Buffer{}
	err := pprof.Lookup("goroutine").WriteTo(&buf, 1)
	if err != nil {
		return context.String(http.StatusInternalServerError, err.Error())
	}

	return context.Stream(http.StatusOK, "application/octet-stream", &buf)
}

func Trace(context echo.Context) error {
	seconds, err := strconv.Atoi(context.QueryParam("seconds"))
	if err != nil {
		return context.String(http.StatusBadRequest, "seconds must be integer")
	}
	if seconds == 0 {
		seconds = 10
	}

	reader, writer := io.Pipe()
	err = trace.Start(writer)
	if err != nil {
		return context.String(http.StatusInternalServerError, err.Error())
	}
	go util.RunWithLabels(pprof.Labels("name", "pprof-trace"), func() {
		time.Sleep(time.Duration(seconds) * time.Second)
		trace.Stop()
		_ = writer.Close()
	})

	return context.Stream(http.StatusOK, "application/octet-stream", reader)
}
