package metrics

import (
	"sync"
	"time"
)

const (
	defaultBucketInterval = time.Second * 5
)

// LoadReporter is used for reporting the average load of worker over some time period.
// NOTE: It is assumed that the reported work intervals are much shorter than the period.
type LoadReporter struct {
	Period         time.Duration
	BucketInterval time.Duration

	recentTimes []time.Time
	recentWork  []time.Duration
	sumWork     time.Duration
	mu          sync.Mutex
}

func NewLoadReporter(period time.Duration) *LoadReporter {
	return &LoadReporter{
		Period:         period,
		BucketInterval: defaultBucketInterval,
	}
}

func (r *LoadReporter) ReportWork(d time.Duration) {
	now := time.Now()
	threshold := now.Add(-r.Period)
	bucketThreshold := now.Add(-r.BucketInterval)

	r.mu.Lock()
	defer r.mu.Unlock()
	r.cleanup(threshold)
	l := len(r.recentTimes)
	// Sum the work in the last bucket.
	if l > 0 && r.recentTimes[l-1].After(bucketThreshold) {
		r.recentWork[l-1] += d
	} else {
		r.recentTimes = append(r.recentTimes, now)
		r.recentWork = append(r.recentWork, d)
	}
	r.sumWork += d
}

func (r *LoadReporter) GetAverage() float64 {
	threshold := time.Now().Add(-r.Period)

	r.mu.Lock()
	defer r.mu.Unlock()
	r.cleanup(threshold)
	ret := float64(r.sumWork) / float64(r.Period)
	// If the worker reported a longer work duration (e.g. longer than the period) the quotient can be greater than 1.0,
	// clamp it (see NOTE that work intervals should be shorter than the period).
	if ret > 1.0 {
		ret = 1.0
	}
	return ret
}

func (r *LoadReporter) cleanup(threshold time.Time) {
	for i, t := range r.recentTimes {
		if t.After(threshold) {
			// NOTE: Yes, the slice is not the best queue implementation due to reallocation and copying in stable case,
			// but this is not that important here due to bucketing.
			r.recentTimes = r.recentTimes[i:]
			r.recentWork = r.recentWork[i:]
			return
		}
		r.sumWork -= r.recentWork[i]
	}
	// This happens when the worker becomes idle.
	r.recentTimes = nil
	r.recentWork = nil
}
