package metrics

import (
	"sync"

	"go.uber.org/atomic"
)

const (
	requestHistogramCount = 3
	maxRequestTime        = 30000
)

var (
	IncomingRps                    atomic.Uint64
	IncomingParsingErrors          atomic.Uint64
	OutgoingRps                    atomic.Uint64
	OutgoingErrors                 atomic.Uint64
	DroppedByQueueTimeout          atomic.Uint64
	DroppedByPushAttempts          atomic.Uint64
	DroppedByRetireTime            atomic.Uint64
	QueueLen                       MaxValue
	MaxConcurrentConnections       MaxValue
	MaxConcurrentRequests          MaxValue
	TimeoutedConcurrentConnections atomic.Uint64
	RejectedConcurrentConnections  atomic.Uint64
	RejectedConcurrentRequests     atomic.Uint64
	RequestTimeMillis              *RequestHistogram
	RequestSizeMillis              *RequestHistogram

	ChQueueLen               MaxValue
	ChQueueMemorySize        MaxValue
	ChBatcherMemorySize      MaxValue
	ChCommits                atomic.Int64
	ChInsertedRows           atomic.Int64
	ChFlushedDueToPressure   atomic.Int64
	ChDroppedDueToFullQueue  atomic.Int64
	ChDroppedDueToFullMemory atomic.Int64
	ChDroppedAfterRetries    atomic.Int64
	ChFailedRetries          atomic.Int64
	ChFailuresDuringCleanup  atomic.Int64
	ChDroppedPartitions      atomic.Int64

	chTableSizes map[string]int64
	chTableRows  map[string]int64
	chTablesMu   sync.Mutex

	chWorkerLoads []float64
	chWorkerMu    sync.Mutex

	S3QueueLen               MaxValue
	S3QueueMemorySize        MaxValue
	S3BatcherMemorySize      MaxValue
	S3Writes                 atomic.Int64
	S3PartWrites             atomic.Int64
	S3WrittenBytes           atomic.Int64
	S3FlushedDueToPressure   atomic.Int64
	S3DroppedDueToFullQueue  atomic.Int64
	S3DroppedDueToFullMemory atomic.Int64
	S3DroppedAfterRetries    atomic.Int64
	S3FailedRetries          atomic.Int64
	S3FailuresDuringMerge    atomic.Int64
	S3Merges                 atomic.Int64
	S3Deletions              atomic.Int64
	S3MergedObjects          atomic.Int64
	S3MergerLoad             atomic.Float64

	s3FolderSizes   map[string]int64
	s3FolderObjects map[string]int64
	s3FoldersMu     sync.Mutex

	s3WorkerLoads []float64
	s3WorkerMu    sync.Mutex

	KinesisQueueLen              MaxValue
	KinesisRecords               atomic.Int64
	KinesisErrors                atomic.Int64
	KinesisDroppedDueToFullQueue atomic.Int64
	KinesisRequestTimeMillis     *RequestHistogram

	kinesisWorkerLoads []float64
	kinesisWorkerMu    sync.Mutex

	syslogMessages map[string]*atomic.Int64 // total messages count (ok and error) per syslog server
	syslogErrors   map[string]*atomic.Int64 // error count per syslog server
)

func init() {
	RequestTimeMillis = NewRequestHistogram()
	RequestSizeMillis = NewRequestHistogram()
	KinesisRequestTimeMillis = NewRequestHistogram()
	syslogMessages = map[string]*atomic.Int64{}
	syslogErrors = map[string]*atomic.Int64{}
}

func SetChTableSizes(sizes map[string]int64) {
	chTablesMu.Lock()
	defer chTablesMu.Unlock()
	chTableSizes = sizes
}

func getChTableSizes() map[string]int64 {
	chTablesMu.Lock()
	defer chTablesMu.Unlock()
	return chTableSizes
}

func SetChTableRows(rows map[string]int64) {
	chTablesMu.Lock()
	defer chTablesMu.Unlock()
	chTableRows = rows
}

func getChTableRows() map[string]int64 {
	chTablesMu.Lock()
	defer chTablesMu.Unlock()
	return chTableRows
}

func SetChWorkerLoads(loads []float64) {
	chWorkerMu.Lock()
	defer chWorkerMu.Unlock()
	chWorkerLoads = loads
}

func getChWorkerLoads() []float64 {
	chWorkerMu.Lock()
	defer chWorkerMu.Unlock()
	return chWorkerLoads
}

func SetS3FolderSizes(sizes map[string]int64) {
	s3FoldersMu.Lock()
	defer s3FoldersMu.Unlock()
	s3FolderSizes = sizes
}

func getS3FolderSizes() map[string]int64 {
	s3FoldersMu.Lock()
	defer s3FoldersMu.Unlock()
	return s3FolderSizes
}

func SetS3FolderObjects(counts map[string]int64) {
	s3FoldersMu.Lock()
	defer s3FoldersMu.Unlock()
	s3FolderObjects = counts
}

func getS3FolderObjects() map[string]int64 {
	s3FoldersMu.Lock()
	defer s3FoldersMu.Unlock()
	return s3FolderObjects
}

func SetS3WorkerLoads(loads []float64) {
	s3WorkerMu.Lock()
	defer s3WorkerMu.Unlock()
	s3WorkerLoads = loads
}

func getS3WorkerLoads() []float64 {
	s3WorkerMu.Lock()
	defer s3WorkerMu.Unlock()
	return s3WorkerLoads
}

func SetKinesisWorkerLoads(loads []float64) {
	kinesisWorkerMu.Lock()
	defer kinesisWorkerMu.Unlock()
	kinesisWorkerLoads = loads
}

func getKinesisWorkerLoads() []float64 {
	kinesisWorkerMu.Lock()
	defer kinesisWorkerMu.Unlock()
	return kinesisWorkerLoads
}

func InitializeSyslogCounts(name string) {
	syslogMessages[name] = new(atomic.Int64)
	syslogErrors[name] = new(atomic.Int64)
}

func IncrementSyslogMessageCount(name string) {
	syslogMessages[name].Inc()
}

func IncrementSyslogErrorCount(name string) {
	syslogErrors[name].Inc()
}

func getSyslogMessageCounts() map[string]int64 {
	return mapAtomicToInt64(syslogMessages)
}

func getSyslogErrorCounts() map[string]int64 {
	return mapAtomicToInt64(syslogErrors)
}

func sumSizes(sizes map[string]int64) int64 {
	result := int64(0)
	for _, value := range sizes {
		result += value
	}
	return result
}

func computeAverageMax(values []float64) (float64, float64) {
	if len(values) == 0 {
		return 0.0, 0.0
	}
	max := values[0]
	average := 0.0
	for _, v := range values {
		if v > max {
			max = v
		}
		average += v
	}
	average /= float64(len(values))
	return average, max
}

func mapAtomicToInt64(m map[string]*atomic.Int64) map[string]int64 {
	result := make(map[string]int64, len(m))
	for name, count := range m {
		result[name] = count.Load()
	}
	return result
}
