package metrics

import (
	"runtime"
	"strconv"
)

// ReportSolomonJSON outputs in Solomon JSON format https://docs.yandex-team.ru/solomon/data-collection/dataformat/json.
// While there is a library/go/core/metrics available, it is easier to implement serialization ourselves for now.
// In the future the unistat should be dropped and the metrics/ package would be then replaced by the registry.
func ReportSolomonJSON() interface{} {
	var memstats runtime.MemStats
	runtime.ReadMemStats(&memstats)

	req50, req95, req99 := RequestTimeMillis.StandardQuantiles()
	size50, size95, size99 := RequestSizeMillis.StandardQuantiles()
	chTableSizes := getChTableSizes()
	chTotalSize := sumSizes(chTableSizes)
	chTableRows := getChTableRows()
	chTotalRows := sumSizes(chTableRows)
	s3FolderSizes := getS3FolderSizes()
	s3TotalSize := sumSizes(s3FolderSizes)
	s3FolderObjects := getS3FolderObjects()
	s3TotalObjects := sumSizes(s3FolderObjects)
	syslogMessages := getSyslogMessageCounts()
	syslogTotalMessages := sumSizes(syslogMessages)
	syslogErrors := getSyslogErrorCounts()
	syslogTotalErrors := sumSizes(syslogErrors)

	chLoads := getChWorkerLoads()
	chWorkerAverage, chWorkerMax := computeAverageMax(chLoads)
	s3Loads := getS3WorkerLoads()
	s3WorkerAverage, s3WorkerMax := computeAverageMax(s3Loads)
	kinesisLoads := getKinesisWorkerLoads()
	kinesisWorkerAverage, _ := computeAverageMax(kinesisLoads)
	kinesisReq50, kinesisReq95, kinesisReq99 := KinesisRequestTimeMillis.StandardQuantiles()

	metrics := []interface{}{
		makeGaugeMetric("alive", 1),
		makeGaugeMetric("num_goroutines", int64(runtime.NumGoroutine())),
		makeGaugeMetricU("heap_size", memstats.HeapAlloc),
		makeGaugeMetricU("heap_objects", memstats.HeapObjects),

		makeRateMetricU("rps-incoming", IncomingRps.Load()),
		makeRateMetricU("rps-outgoing", OutgoingRps.Load()),
		makeGaugeMetric("request-time-p50", req50),
		makeGaugeMetric("request-time-p95", req95),
		makeGaugeMetric("request-time-p99", req99),
		makeGaugeMetric("request-size-p50", size50),
		makeGaugeMetric("request-size-p95", size95),
		makeGaugeMetric("request-size-p99", size99),
		makeGaugeMetricU("concurrent-connections", MaxConcurrentConnections.Get()),
		makeGaugeMetricU("concurrent-requests", MaxConcurrentRequests.Get()),
		makeRateMetricU("timeouted-concurrent-connections", TimeoutedConcurrentConnections.Load()),
		makeRateMetricU("rejected-concurrent-connections", RejectedConcurrentConnections.Load()),
		makeRateMetricU("rejected-concurrent-requests", RejectedConcurrentRequests.Load()),
		makeRateMetricU("errors-incoming-parsing", IncomingParsingErrors.Load()),
		makeRateMetricU("errors-outgoing", OutgoingErrors.Load()),
		makeRateMetricU("errors-dropped-queue-timeout", DroppedByQueueTimeout.Load()),
		makeRateMetricU("errors-dropped-push-attempts", DroppedByPushAttempts.Load()),
		makeRateMetricU("errors-dropped-retire-time", DroppedByRetireTime.Load()),
		makeGaugeMetricU("len-queue", QueueLen.Get()),

		makeGaugeMetricU("ch-len-queue", ChQueueLen.Get()),
		makeGaugeMetricU("ch-size-queue", ChQueueMemorySize.Get()),
		makeGaugeMetricU("ch-size-batcher", ChBatcherMemorySize.Get()),
		makeRateMetric("ch-commits", ChCommits.Load()),
		makeRateMetric("ch-inserted-rows", ChInsertedRows.Load()),
		makeGaugeMetricF("ch-average-worker-load", chWorkerAverage),
		makeGaugeMetricF("ch-max-worker-load", chWorkerMax),
		makeRateMetric("ch-flushed-pressure", ChFlushedDueToPressure.Load()),
		makeRateMetric("ch-dropped-queue-full", ChDroppedDueToFullQueue.Load()),
		makeRateMetric("ch-dropped-memory-full", ChDroppedDueToFullMemory.Load()),
		makeRateMetric("ch-dropped-after-retries", ChDroppedAfterRetries.Load()),
		makeRateMetric("ch-failed-retries", ChFailedRetries.Load()),
		makeRateMetric("ch-failures-during-cleanup", ChFailuresDuringCleanup.Load()),
		makeRateMetric("ch-dropped-partitions", ChDroppedPartitions.Load()),
		makeGaugeMetric("ch-total-size", chTotalSize),
		makeGaugeMetric("ch-total-rows", chTotalRows),

		makeGaugeMetricU("s3-len-queue", S3QueueLen.Get()),
		makeGaugeMetricU("s3-size-queue", S3QueueMemorySize.Get()),
		makeGaugeMetricU("s3-size-batcher", S3BatcherMemorySize.Get()),
		makeRateMetric("s3-writes", S3Writes.Load()),
		makeRateMetric("s3-part-writes", S3PartWrites.Load()),
		makeRateMetric("s3-written-bytes", S3WrittenBytes.Load()),
		makeGaugeMetricF("s3-average-worker-load", s3WorkerAverage),
		makeGaugeMetricF("s3-max-worker-load", s3WorkerMax),
		makeRateMetric("s3-flushed-pressure", S3FlushedDueToPressure.Load()),
		makeRateMetric("s3-dropped-queue-full", S3DroppedDueToFullQueue.Load()),
		makeRateMetric("s3-dropped-memory-full", S3DroppedDueToFullMemory.Load()),
		makeRateMetric("s3-dropped-after-retries", S3DroppedAfterRetries.Load()),
		makeRateMetric("s3-failed-retries", S3FailedRetries.Load()),
		makeRateMetric("s3-failures-during-merge", S3FailuresDuringMerge.Load()),
		makeRateMetric("s3-merges", S3Merges.Load()),
		makeRateMetric("s3-deletions", S3Deletions.Load()),
		makeGaugeMetric("s3-total-size", s3TotalSize),
		makeGaugeMetric("s3-total-objects", s3TotalObjects),
		makeGaugeMetric("s3-merged-objects", S3MergedObjects.Load()),
		makeGaugeMetricF("s3-merger-load", S3MergerLoad.Load()),

		makeGaugeMetricU("kinesis-len-queue", KinesisQueueLen.Get()),
		makeRateMetric("kinesis-records", KinesisRecords.Load()),
		makeGaugeMetric("kinesis-request-time-p50", kinesisReq50),
		makeGaugeMetric("kinesis-request-time-p95", kinesisReq95),
		makeGaugeMetric("kinesis-request-time-p99", kinesisReq99),
		makeRateMetric("kinesis-errors", KinesisErrors.Load()),
		makeRateMetric("kinesis-dropped-queue-full", KinesisDroppedDueToFullQueue.Load()),
		makeGaugeMetricF("kinesis-average-worker-load", kinesisWorkerAverage),

		makeRateMetric("syslog-total-messages", syslogTotalMessages),
		makeRateMetric("syslog-total-errors", syslogTotalErrors),
	}
	for table, size := range chTableSizes {
		metrics = append(metrics, makeGaugeMetricWithLabel("ch-table-size", size, "table", table))
	}
	for table, count := range chTableRows {
		metrics = append(metrics, makeGaugeMetricWithLabel("ch-table-rows", count, "table", table))
	}
	for folder, size := range s3FolderSizes {
		metrics = append(metrics, makeGaugeMetricWithLabel("s3-folder-size", size, "folder", folder))
	}
	for folder, count := range s3FolderObjects {
		metrics = append(metrics, makeGaugeMetricWithLabel("s3-folder-objects", count, "folder", folder))
	}
	for i, load := range chWorkerLoads {
		metrics = append(metrics, makeGaugeMetricWithLabelF("ch-worker-load", load, "worker", strconv.Itoa(i)))
	}
	for i, load := range s3WorkerLoads {
		metrics = append(metrics, makeGaugeMetricWithLabelF("s3-worker-load", load, "worker", strconv.Itoa(i)))
	}
	for name, count := range syslogMessages {
		metrics = append(metrics, makeRateMetricWithLabel("syslog-messages", count, "parser_name", name))
	}
	for name, count := range syslogErrors {
		metrics = append(metrics, makeRateMetricWithLabel("syslog-errors", count, "parser_name", name))
	}

	return map[string]interface{}{
		"metrics": metrics,
	}
}

func makeGaugeMetric(name string, value int64) map[string]interface{} {
	return map[string]interface{}{
		"type": "GAUGE",
		"labels": map[string]interface{}{
			"sensor": name,
		},
		"value": value,
	}
}

func makeGaugeMetricU(name string, value uint64) map[string]interface{} {
	return makeGaugeMetric(name, int64(value))
}

func makeGaugeMetricF(name string, value float64) map[string]interface{} {
	return map[string]interface{}{
		"type": "GAUGE",
		"labels": map[string]interface{}{
			"sensor": name,
		},
		"value": value,
	}
}

func makeGaugeMetricWithLabel(name string, value int64, labelName string, labelValue string) map[string]interface{} {
	return map[string]interface{}{
		"type": "GAUGE",
		"labels": map[string]interface{}{
			"sensor":  name,
			labelName: labelValue,
		},
		"value": value,
	}
}

func makeGaugeMetricWithLabelF(name string, value float64, labelName string, labelValue string) interface{} {
	return map[string]interface{}{
		"type": "GAUGE",
		"labels": map[string]interface{}{
			"sensor":  name,
			labelName: labelValue,
		},
		"value": value,
	}
}

func makeRateMetric(name string, value int64) map[string]interface{} {
	return map[string]interface{}{
		"type": "RATE",
		"labels": map[string]interface{}{
			"sensor": name,
		},
		"value": value,
	}
}

func makeRateMetricU(name string, value uint64) map[string]interface{} {
	return makeRateMetric(name, int64(value))
}

func makeRateMetricWithLabel(name string, value int64, labelName string, labelValue string) map[string]interface{} {
	return map[string]interface{}{
		"type": "RATE",
		"labels": map[string]interface{}{
			"sensor":  name,
			labelName: labelValue,
		},
		"value": value,
	}
}
