package parser

import (
	"fmt"
	"log"
	"net"

	"a.yandex-team.ru/security/osquery/osquery-sender/config"
)

// LogMessage is the struct of message sent by osquery logger plugin,
// see https://github.com/osquery/osquery/blob/master/plugins/logger/tls_logger.cpp
//
// This log format is a container for the actual log data sent e.g. via Kinesis plugin (see described in
// https://osquery.readthedocs.io/en/stable/deployment/logging/#schedule-results
//
// See also https://st.yandex-team.ru/CLOUD-81620
type LogMessage struct {
	LogType string `json:"log_type"`
	NodeKey string `json:"node_key"`
	// Data contains an array of records, described in https://osquery.readthedocs.io/en/stable/deployment/logging/#schedule-results
	// (see batcher for parsing details).
	Data []map[string]interface{} `json:"data"`
}

func ParseIntoEvents(message *LogMessage, peerIP net.IP, conf *config.SenderConfig) []*ParsedEvent {
	events := make([]*ParsedEvent, 0, len(message.Data))

	for _, item := range message.Data {
		var host interface{}
		var ok bool
		for _, key := range conf.HostnameKeys {
			host, ok = item[key]
			if ok {
				break
			}
		}
		if host == nil {
			log.Printf("ERROR: invalid format - cannot get value for %v\n", conf.HostnameKeys)
			continue
		}

		hostStr, ok := host.(string)
		if !ok {
			log.Printf("ERROR: invalid format - cannot cast value for %v to string\n", conf.HostnameKeys)
			continue
		}

		name, ok := item["name"]
		if !ok {
			name = ""
		}
		nameStr, ok := name.(string)
		if !ok {
			log.Println("ERROR: invalid format - cannot cast 'name' to string")
			continue
		}

		addForPeer(conf.AddForPeers, peerIP, item)

		event := &ParsedEvent{
			Host:    hostStr,
			LogType: message.LogType,
			NodeKey: message.NodeKey,
			Name:    nameStr,
			Data:    item,
		}
		events = append(events, event)
	}

	if len(events) == 0 {
		return nil
	}

	if conf.EnableDebug {
		printEvents(events)
	}

	return events
}

func addForPeer(configs []config.AddForPeersConfig, ip net.IP, item map[string]interface{}) {
	for _, c := range configs {
		for _, subnet := range c.Subnets {
			if subnet.Contains(ip) {
				for k, v := range c.Values {
					item[k] = v
				}
				return
			}
		}
	}
}

func printEvents(events []*ParsedEvent) {
	const printableSize = 900
	eventsStr := fmt.Sprint(events)
	for len(eventsStr) > printableSize {
		log.Println(eventsStr[:printableSize])
		eventsStr = eventsStr[printableSize:]
	}
	log.Println(eventsStr)
}
