package s3

import (
	"fmt"
	"log"
	"runtime/debug"
	"runtime/pprof"

	"a.yandex-team.ru/security/osquery/osquery-sender/util"
)

// JobPool is a basic job pool with a fixed number of workers.
type JobPool struct {
	name string
	jobs chan *jobPair
}

type Job func() (interface{}, error)

type Future struct {
	result chan interface{}
	err    chan error
}

func (f *Future) Get() (interface{}, error) {
	select {
	case result := <-f.result:
		return result, nil
	case err := <-f.err:
		return nil, err
	}
}

func NewThreadPool(size int, queueSize int, name string) *JobPool {
	ret := &JobPool{
		name: name,
		jobs: make(chan *jobPair, queueSize),
	}
	for i := 0; i < size; i++ {
		go ret.workerRun()
	}
	return ret
}

type jobPair struct {
	job    Job
	future Future
}

func (p *JobPool) Submit(job Job) *Future {
	future := Future{
		result: make(chan interface{}, 1),
		err:    make(chan error, 1),
	}
	pair := &jobPair{
		job:    job,
		future: future,
	}
	p.jobs <- pair
	return &future
}

func (p *JobPool) SubmitWithLabels(labels pprof.LabelSet, job Job) *Future {
	return p.Submit(func() (interface{}, error) {
		var ret interface{}
		var err error
		util.RunWithLabels(labels, func() {
			ret, err = job()
		})
		return ret, err
	})
}

func (p *JobPool) Close() {
	close(p.jobs)
}

func (p *JobPool) workerRun() {
	for job := range p.jobs {
		func() {
			defer func() {
				if r := recover(); r != nil {
					log.Println("ERROR: recovered in "+p.name+" worker", r, string(debug.Stack()))
					job.future.err <- fmt.Errorf("internal error in %s worker: %v", p.name, r)
				}
			}()

			result, err := job.job()
			if err != nil {
				job.future.err <- err
			} else {
				job.future.result <- result
			}
		}()
	}
}
