package s3

import (
	"errors"
	"fmt"
	"math"
	"math/rand"
	"strconv"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws/awserr"
)

type CompressionAlg int

const (
	CompressionNone = CompressionAlg(iota)
	CompressionLz4
	CompressionSnappy
)

const (
	// AWS Go SDK canonicalizes metadata keys, use pre-canonicalized names, otherwise we will not be able to read it back.
	batchTimeMeta   = "Batch_time"
	compressionMeta = "Compression_alg"
	columnsMeta     = "Columns"
	dateMeta        = "Date"

	keyPattern = "2006-01-02T15-04-05"
)

func ParseCompressionAlg(s string) (CompressionAlg, error) {
	switch strings.ToLower(s) {
	case "":
		return defaultCompressionAlg, nil
	case "none":
		return CompressionNone, nil
	case "lz4":
		return CompressionLz4, nil
	case "snappy":
		return CompressionSnappy, nil
	default:
		return CompressionNone, fmt.Errorf("unknown compression algorithm %s", s)
	}
}

func (alg CompressionAlg) String() string {
	switch alg {
	case CompressionNone:
		return ""
	case CompressionLz4:
		return "lz4"
	case CompressionSnappy:
		return "snappy"
	default:
		return "unknown"
	}
}

func getCompressionAlgSuffix(alg CompressionAlg) string {
	switch alg {
	case CompressionNone:
		return ""
	default:
		return "." + alg.String()
	}
}

func MakeKey(name string, batchTime time.Time, loc *time.Location, alg CompressionAlg) string {
	timestamp := batchTime.In(loc).Format(keyPattern)
	return fmt.Sprintf("/%s/%s-%d.tsv%s", name, timestamp, rand.Int63(), getCompressionAlgSuffix(alg))
}

func MakeMergedKey(name string, date int, alg CompressionAlg) string {
	year := date / 10000
	month := (date / 100) % 100
	day := date % 100
	return fmt.Sprintf("/%s/merged/%d-%02d-%02d.tsv%s", name, year, month, day, getCompressionAlgSuffix(alg))
}

func parseNameFromKey(key string) (string, error) {
	if key == "" {
		return "", errors.New("empty key")
	}
	// Remove leading slash
	if key[0] == '/' {
		key = key[1:]
	}
	v := strings.Split(key, "/")
	if !isTsvKey(v) {
		return "", fmt.Errorf("cannot parse key: %s", key)
	}
	return v[0], nil
}

// Either packname/file.tsv or packname/merged/file.tsv
func isTsvKey(v []string) bool {
	if len(v) != 2 && len(v) != 3 {
		return false
	}
	if !strings.Contains(v[len(v)-1], ".tsv") {
		return false
	}
	return true
}

func isMergedKey(key string) bool {
	return strings.Contains(key, "/merged/")
}

func MakeMetadata(batchTime time.Time, alg CompressionAlg, columns []string) map[string]*string {
	strTime := strconv.FormatInt(batchTime.Unix(), 10)
	strAlg := alg.String()
	strColumns := strings.Join(columns, "\t")
	return map[string]*string{
		batchTimeMeta:   &strTime,
		compressionMeta: &strAlg,
		columnsMeta:     &strColumns,
	}
}

func MakeMergedMetadata(date int, alg CompressionAlg, columns []string) map[string]*string {
	strDate := strconv.Itoa(date)
	strAlg := alg.String()
	strColumns := strings.Join(columns, "\t")
	return map[string]*string{
		dateMeta:        &strDate,
		compressionMeta: &strAlg,
		columnsMeta:     &strColumns,
	}
}

func parseMetadataBatchTime(metadata map[string]*string) (time.Time, error) {
	strTime, ok := metadata[batchTimeMeta]
	if !ok {
		return time.Time{}, errors.New("no " + batchTimeMeta + " in metadata")
	}
	timestamp, err := strconv.ParseInt(*strTime, 10, 64)
	if err != nil {
		return time.Time{}, err
	}
	return time.Unix(timestamp, 0), nil
}

func ParseMetadataAlg(metadata map[string]*string) (CompressionAlg, error) {
	strAlg, ok := metadata[compressionMeta]
	if !ok {
		return CompressionNone, errors.New("no " + compressionMeta + " in metadata")
	}
	return ParseCompressionAlg(*strAlg)
}

func ParseMetadataColumns(metadata map[string]*string) ([]string, error) {
	strColumns, ok := metadata[columnsMeta]
	if !ok {
		return nil, errors.New("no " + columnsMeta + " in metadata")
	}
	return strings.Split(*strColumns, "\t"), nil
}

func parseMetadataDate(metadata map[string]*string) (int, error) {
	strDate, ok := metadata[dateMeta]
	if !ok {
		return math.MaxInt32, errors.New("no " + dateMeta + " in metadata")
	}
	date, err := strconv.Atoi(*strDate)
	if err != nil {
		return math.MaxInt32, err
	}
	return date, nil
}

func errToString(err error) string {
	if ae, ok := err.(awserr.Error); ok {
		if ae.OrigErr() != nil {
			return fmt.Sprintf("%v caused by %v", err.Error(), ae.OrigErr().Error())
		} else {
			return err.Error()
		}
	} else if ae, ok := err.(awserr.BatchedErrors); ok {
		var messages []string
		for _, e := range ae.OrigErrs() {
			messages = append(messages, e.Error())
		}
		return fmt.Sprintf("%v caused by %v", err.Error(), strings.Join(messages, ", "))
	} else {
		return err.Error()
	}
}
