package s3

import (
	"errors"
	"io"

	"github.com/golang/snappy"
	"github.com/pierrec/lz4"
)

// A wrapper for io.Writer which calculates the amount of written bytes.
type countingWriter struct {
	writer  io.Writer
	written int64
	noClose bool
}

func (w *countingWriter) Write(p []byte) (n int, err error) {
	n, err = w.writer.Write(p)
	w.written += int64(n)
	return
}

func (w *countingWriter) Close() error {
	if w.noClose {
		return nil
	}
	// This is mostly a workaround for bytes.Buffer.
	if closer, ok := w.writer.(io.Closer); ok {
		return closer.Close()
	}
	return nil
}

func NewCompressedWriter(alg CompressionAlg, writer io.Writer) (*countingWriter, error) {
	switch alg {
	case CompressionNone:
		return &countingWriter{
			writer: writer,
			// None of lz4/snappy writers close the underlying writer on Close().
			noClose: true,
		}, nil
	case CompressionLz4:
		lz4Writer := lz4.NewWriter(writer)
		// Reduce the block size a bit, this does not meaningfully affect the compression but reduces the CPU usage.
		lz4Writer.Header.BlockMaxSize = 256 * 1024
		return &countingWriter{writer: lz4Writer}, nil
	case CompressionSnappy:
		return &countingWriter{writer: snappy.NewBufferedWriter(writer)}, nil
	default:
		return nil, errors.New("unsupported compression type " + alg.String())
	}
}

func NewCompressedReader(alg CompressionAlg, r io.Reader) (io.Reader, error) {
	switch alg {
	case CompressionLz4:
		return lz4.NewReader(r), nil
	case CompressionSnappy:
		return snappy.NewReader(r), nil
	case CompressionNone:
		return r, nil
	default:
		return nil, errors.New("unknown compression algorithm " + alg.String())
	}
}
