package s3

import (
	"encoding/csv"
	"fmt"
	"io"
	"strconv"

	"a.yandex-team.ru/security/osquery/osquery-sender/batcher"
)

const (
	timestampField       = "timestamp"
	actionField          = "action"
	hostField            = "host"
	numPredefinedColumns = 3
)

// TsvWriter serializes EventBatch into tab-separated rows, optionally compressing the output stream.
type TsvWriter struct {
	columns []string

	writer           *countingWriter
	compressedWriter *countingWriter
	csvW             *csv.Writer
}

func NewTsvWriter(columns []string, alg CompressionAlg, innerWriter io.WriteCloser) (*TsvWriter, error) {
	compressedWriter := &countingWriter{writer: innerWriter}
	writer, err := NewCompressedWriter(alg, compressedWriter)
	if err != nil {
		return nil, err
	}

	csvW := csv.NewWriter(writer)
	csvW.Comma = '\t'

	ret := &TsvWriter{
		columns: columns,

		writer:           writer,
		compressedWriter: compressedWriter,
		csvW:             csvW,
	}

	err = ret.writeHeader()
	if err != nil {
		return nil, fmt.Errorf("writing TSV header failed: %v", errToString(err))
	}

	return ret, nil
}

func (w *TsvWriter) Write(events *batcher.EventBatch) error {
	record := make([]string, events.NumColumns+numPredefinedColumns)
	for pos := 0; pos < events.Length; pos++ {
		record[0] = strconv.FormatInt(events.Timestamps[pos], 10)
		record[1] = events.Actions[pos]
		record[2] = events.Hosts[pos]
		for i, column := range w.columns {
			if v, ok := events.Float64Values[column]; ok {
				record[numPredefinedColumns+i] = strconv.FormatFloat(v[pos], 'g', -1, 64)
			} else if v, ok := events.StringValues[column]; ok {
				record[numPredefinedColumns+i] = v[pos]
			} else {
				return fmt.Errorf("unknown column type for %s", column)
			}
		}
		err := w.csvW.Write(record)
		if err != nil {
			return err
		}
	}
	return nil
}

func (w *TsvWriter) Close() error {
	w.csvW.Flush()
	return w.writer.Close()
}

func (w *TsvWriter) TotalBytes() int64 {
	return w.writer.written
}

func (w *TsvWriter) CompressedBytes() int64 {
	return w.compressedWriter.written
}

func (w *TsvWriter) writeHeader() error {
	header := make([]string, 0, len(w.columns)+numPredefinedColumns)
	header = append(header, timestampField)
	header = append(header, actionField)
	header = append(header, hostField)
	header = append(header, w.columns...)
	err := w.csvW.Write(header)
	if err != nil {
		return err
	}

	return nil
}

func addPredefinedColumns(columns []string) []string {
	ret := make([]string, len(columns)+numPredefinedColumns)
	ret[0] = timestampField
	ret[1] = actionField
	ret[2] = hostField
	copy(ret[numPredefinedColumns:], columns)
	return ret
}
