package sender

import (
	"strconv"
	"time"

	"a.yandex-team.ru/security/osquery/osquery-sender/enroll"
	"a.yandex-team.ru/security/osquery/osquery-sender/msgconst"
	"a.yandex-team.ru/security/osquery/osquery-sender/parser"
)

type Message struct {
	DstHostname  string
	Data         []EventData
	RetireTime   time.Time
	PushAttempts int
}

func FromEnroll(dstHostname string, nodeKey string, request *enroll.Request) (*Message, error) {
	ev := EventData{
		EventVal: Event{
			Data:    request,
			LogType: "enroll_log",
			NodeKey: nodeKey,
		},
		Host: request.HostIdentifier,
		Time: strconv.FormatInt(time.Now().UTC().Unix(), 10),
	}

	return &Message{
		DstHostname:  dstHostname,
		Data:         []EventData{ev},
		PushAttempts: msgconst.PushAttempts,
		RetireTime:   time.Now().UTC().Add(msgconst.RetireTime),
	}, nil
}

func BatchEventsIntoMessages(dstHostname string, events []*parser.ParsedEvent) []*Message {
	result := make([]*Message, 0, len(events)/msgconst.ChunkSize+1)
	retireTime := time.Now().UTC().Add(msgconst.RetireTime)
	for len(events) > 0 {
		var batch []*parser.ParsedEvent
		if len(events) < msgconst.ChunkSize {
			batch = events
			events = nil
		} else {
			batch = events[:msgconst.ChunkSize]
			events = events[msgconst.ChunkSize:]
		}

		message := &Message{
			DstHostname:  dstHostname,
			Data:         eventsToEventData(batch),
			PushAttempts: msgconst.PushAttempts,
			RetireTime:   retireTime,
		}
		result = append(result, message)
	}
	return result
}

func eventsToEventData(events []*parser.ParsedEvent) []EventData {
	result := make([]EventData, 0, len(events))
	timeNow := strconv.FormatInt(time.Now().UTC().Unix(), 10)
	for _, event := range events {
		var source string
		// Splunk requires source to be stored in root object.
		if sourceI, ok := event.Data["source"]; ok {
			source = sourceI.(string)
		}
		eventData := EventData{
			EventVal: Event{
				Data:    event.Data,
				LogType: event.LogType,
				NodeKey: event.NodeKey,
			},
			Host:   event.Host,
			Time:   timeNow,
			Source: source,
		}
		result = append(result, eventData)
	}
	return result
}
