package sender

import (
	"bytes"
	"crypto/tls"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"runtime/debug"
	"runtime/pprof"
	"strconv"
	"strings"
	"sync"
	"time"

	"go.uber.org/atomic"

	"a.yandex-team.ru/security/osquery/osquery-sender/config"
	"a.yandex-team.ru/security/osquery/osquery-sender/metrics"
	"a.yandex-team.ru/security/osquery/osquery-sender/msgconst"
	"a.yandex-team.ru/security/osquery/osquery-sender/util"
)

type Sender struct {
	queue        chan *Message
	splunkConfig map[string]*config.HostConfig
	workersCount int
	wg           sync.WaitGroup
	client       *http.Client
	retryQueue   chan retryMessage
	stopped      atomic.Bool
}

type retryMessage struct {
	message *Message
	retryIn time.Time
}

func NewSender(hostsConfig map[string]*config.HostConfig, config *config.SplunkConfig, insecure bool) (*Sender, error) {
	// An optimization: do not start Splunk sender if no splunk URLs are configured.
	if allURLAreEmpty(hostsConfig) {
		return nil, nil
	}

	client := &http.Client{
		Timeout: 10 * time.Second,
		Transport: &http.Transport{
			TLSHandshakeTimeout: 5 * time.Second,
			DisableKeepAlives:   false,
			TLSClientConfig:     &tls.Config{InsecureSkipVerify: insecure},
		},
	}

	err := readTokenFiles(hostsConfig)
	if err != nil {
		return nil, err
	}

	return &Sender{
		queue:        make(chan *Message, config.QueueLength),
		workersCount: config.Workers,
		splunkConfig: hostsConfig,
		client:       client,
		retryQueue:   make(chan retryMessage, config.QueueLength*2),
	}, nil
}

func allURLAreEmpty(hostsConfig map[string]*config.HostConfig) bool {
	for _, item := range hostsConfig {
		if item.SplunkURL != "" {
			return false
		}
	}
	return true
}

func readTokenFiles(config map[string]*config.HostConfig) error {
	for hostname, item := range config {
		if item.SplunkTokenFile != "" {
			if item.SplunkToken != "" {
				return fmt.Errorf("config for %s has both token and token_file", hostname)
			}
			contents, err := ioutil.ReadFile(item.SplunkTokenFile)
			if err != nil {
				return err
			}
			item.SplunkToken = strings.TrimSpace(string(contents))
		}
	}
	return nil
}

func (s *Sender) Start() {
	for i := 0; i < s.workersCount; i++ {
		s.wg.Add(1)

		i := i
		go util.RunWithLabels(pprof.Labels("name", "sender-worker-"+strconv.Itoa(i)), func() {
			defer s.wg.Done()

			for msg := range s.queue {
				if msg == nil {
					continue
				}

				s.sendMessage(msg, true)
			}

			for msg := range s.retryQueue {
				s.sendMessage(msg.message, false)
			}
		})
	}
	go util.RunWithLabels(pprof.Labels("name", "sender-retry-mover"), func() {
		s.retryMover()
	})
}

func (s *Sender) Push(message *Message) {
	s.wg.Add(1)
	defer func() {
		s.wg.Done()
		if r := recover(); r != nil {
			log.Println("ERROR: recovered in pushMsgToSender ", r, string(debug.Stack()))
		}
	}()

	select {
	case s.queue <- message:
		s.updateMetricsMaxValues()
	case <-time.After(msgconst.InsertTimeout):
		log.Println("ERROR: queue is full: dropping request not putting it on queue")
		metrics.DroppedByQueueTimeout.Inc()
	}
}

func (s *Sender) Stop() {
	close(s.queue)
	close(s.retryQueue)
	s.wg.Wait()
	s.stopped.Store(true)
}

func (s *Sender) UpdateMetrics() {
	s.updateMetricsMaxValues()
}

func (s *Sender) updateMetricsMaxValues() {
	metrics.QueueLen.Report(uint64(len(s.queue)))
}

func (s *Sender) rescheduleMessage(message *Message) {
	if message.RetireTime.Before(time.Now().UTC()) {
		metrics.DroppedByRetireTime.Inc()
		return
	}
	message.PushAttempts--
	if message.PushAttempts == 0 {
		metrics.DroppedByPushAttempts.Inc()
		return
	}

	retryIn := time.Now().Add(msgconst.RescheduleSleepTime)
	select {
	case s.retryQueue <- retryMessage{message: message, retryIn: retryIn}:
	default:
		log.Println("ERROR: retry queue is full, dropping request during reschedule")
		metrics.DroppedByQueueTimeout.Inc()
		return
	}
}

func (s *Sender) sendMessage(msg *Message, withRetry bool) {
	data, err := json.Marshal(msg.Data)
	if err != nil {
		log.Printf("error: cannot serialize data: %v \n", err)
		metrics.OutgoingErrors.Inc()
		return
	}
	splunkConfig := s.splunkConfig[msg.DstHostname]
	if splunkConfig == nil {
		log.Printf("error: splunk config not found for hostname: %v", msg.DstHostname)
		metrics.OutgoingErrors.Inc()
		return
	}
	if splunkConfig.SplunkURL == "" {
		return
	}
	req, err := http.NewRequest("POST", splunkConfig.SplunkURL, bytes.NewBuffer(data))
	if err != nil {
		log.Printf("error: cannot send request: %v \n", err)
		metrics.OutgoingErrors.Inc()
		return
	}

	req.Header.Set("Authorization", fmt.Sprintf("Splunk %s", splunkConfig.SplunkToken))
	req.Header.Set("Content-Type", "application/json")

	metrics.OutgoingRps.Inc()
	if _, err := s.client.Do(req); err != nil {
		log.Printf("error: cannot send request: %v \n", err)
		metrics.OutgoingErrors.Inc()
		if withRetry {
			s.rescheduleMessage(msg)
		}
	}
}

func (s *Sender) retryMover() {
	for {
		if s.stopped.Load() {
			return
		}

		now := time.Now()
		var tooYoung []retryMessage
		for len(s.retryQueue) > 0 {
			toRetry := <-s.retryQueue
			if toRetry.retryIn.After(now) {
				s.queue <- toRetry.message
			} else {
				tooYoung = append(tooYoung, toRetry)
			}
		}

		for _, m := range tooYoung {
			select {
			case s.retryQueue <- m:
			default:
				log.Println("ERROR: retry queue is full, dropping request during reschedule (in retryMover)")
				metrics.DroppedByQueueTimeout.Inc()
				return
			}
		}

		time.Sleep(msgconst.RescheduleSleepTime)
	}
}
