package sendmgr

import (
	"runtime/pprof"

	"a.yandex-team.ru/security/osquery/osquery-sender/clickhouse"
	"a.yandex-team.ru/security/osquery/osquery-sender/config"
	"a.yandex-team.ru/security/osquery/osquery-sender/enroll"
	"a.yandex-team.ru/security/osquery/osquery-sender/kinesis"
	"a.yandex-team.ru/security/osquery/osquery-sender/parser"
	"a.yandex-team.ru/security/osquery/osquery-sender/s3"
	"a.yandex-team.ru/security/osquery/osquery-sender/sender"
	"a.yandex-team.ru/security/osquery/osquery-sender/util"
)

type SendMgr struct {
	sndr        *sender.Sender
	chSndr      *clickhouse.ClickhouseSender
	s3Sndr      *s3.S3Sender
	kinesisSndr *kinesis.KinesisSender
	enableDebug bool
}

func NewSendMgr(conf *config.SenderConfig) *SendMgr {
	sndr, err := sender.NewSender(
		conf.HostsConfig,
		conf.Splunk,
		conf.TLSInsecure,
	)
	if err != nil {
		panic(err)
	}

	addDecorators := getAddDecorators(conf)

	chSndr, err := clickhouse.NewSender(conf.Clickhouse, addDecorators, conf.EnableDebug)
	if err != nil {
		panic(err)
	}

	s3Sndr, err := s3.NewSender(conf.S3, addDecorators, conf.EnableDebug)
	if err != nil {
		panic(err)
	}

	kinesisSndr, err := kinesis.NewSender(conf.Kinesis, conf.EnableDebug)
	if err != nil {
		panic(err)
	}

	if sndr != nil {
		sndr.Start()
	}
	if chSndr != nil {
		chSndr.Start()
	}
	if s3Sndr != nil {
		s3Sndr.Start()
	}
	if kinesisSndr != nil {
		kinesisSndr.Start()
	}
	return &SendMgr{
		sndr:        sndr,
		chSndr:      chSndr,
		s3Sndr:      s3Sndr,
		kinesisSndr: kinesisSndr,
		enableDebug: conf.EnableDebug,
	}
}

func getAddDecorators(conf *config.SenderConfig) []string {
	result := conf.AddDecorators
	if len(conf.AddForPeers) > 0 {
		// We must add all keys from AddForPeers to AddDecorators, otherwise batcher code will ignore those new values.
		addedKeys := make(map[string]bool)
		for _, c := range conf.AddForPeers {
			for k := range c.Values {
				addedKeys[k] = true
			}
		}

		for k := range addedKeys {
			result = append(result, k)
		}
	}
	return result
}

func (s *SendMgr) UpdateMetrics() {
	if s.sndr != nil {
		s.sndr.UpdateMetrics()
	}
	if s.chSndr != nil {
		s.chSndr.UpdateMetrics()
	}
	if s.s3Sndr != nil {
		s.s3Sndr.UpdateMetrics()
	}
	if s.kinesisSndr != nil {
		s.kinesisSndr.UpdateMetrics()
	}
}

func (s *SendMgr) Log(dstHostname string, events []*parser.ParsedEvent) {
	if s.s3Sndr != nil {
		s.s3Sndr.Enqueue(events)
	}

	var otherEvents []*parser.ParsedEvent
	if s.chSndr != nil {
		var chEvents []*parser.ParsedEvent
		chEvents, otherEvents = s.chSndr.FilterEvents(events)
		s.chSndr.Enqueue(chEvents)
	} else {
		otherEvents = events
	}

	if s.sndr != nil {
		msgs := sender.BatchEventsIntoMessages(dstHostname, otherEvents)
		for _, msg := range msgs {
			msg := msg
			go util.RunWithLabels(pprof.Labels("name", "sender-push-msg", "host", dstHostname), func() {
				s.sndr.Push(msg)
			})
		}
	}

	if s.kinesisSndr != nil {
		s.kinesisSndr.Send(events)
	}
}

func (s *SendMgr) LogEnroll(dstHostname string, nodeKey string, request *enroll.Request) {
	if msg, err := sender.FromEnroll(dstHostname, nodeKey, request); err == nil {
		go util.RunWithLabels(pprof.Labels("name", "sender-enroll-push-msg", "host", dstHostname), func() {
			s.sndr.Push(msg)
		})
	}
}

func (s *SendMgr) Stop() {
	if s.sndr != nil {
		s.sndr.Stop()
	}
	if s.chSndr != nil {
		s.chSndr.Stop()
	}
	if s.s3Sndr != nil {
		s.s3Sndr.Stop()
	}
	if s.kinesisSndr != nil {
		s.kinesisSndr.Stop()
	}
}
