package formats

import (
	"fmt"
	"regexp"
	"time"

	"gopkg.in/mcuadros/go-syslog.v2/format"
)

// RegexFormat implements SimpleFormat.
// It parses logs using a Regex with named groups,
// so the resulting LogParts will contain values indexed by those group names.
// LogParts will also contain "content", with the whole log line (this is also done by the formats
// that come with the go-syslog.v2 package).
// GroupParsers can be used for further parsing of LogParts. It will contain a map of name : GroupParser,
// to further parse each logParts[name] with the respective GroupParser.
type RegexFormat struct {
	Regex        *regexp.Regexp
	GroupParsers GroupParsers
}

// FormatFromRegex is a convenience function, since we often use regex formats
func FormatFromRegex(regex *regexp.Regexp, groupParsers GroupParsers) format.Format {
	return &FormatAdapter{
		SimpleFormat: &RegexFormat{Regex: regex, GroupParsers: groupParsers},
	}
}

// DateParserToGroupParsers returns a GroupParsers with just one GroupParser that parses dates.
// The date is expected in logParts["Date"], and the resulting time in set in logParts["Time"].
// You may use this function and then add additional GroupParser(s).
func DateParserToGroupParsers(dateParser DateParser) GroupParsers {
	return map[string]GroupParser{
		"Date": dateParserToGroupParser(dateParser),
	}
}

func dateParserToGroupParser(dateParser DateParser) GroupParser {
	return func(date string, logParts format.LogParts) (err error) {
		t, err := dateParser(date)
		if err != nil {
			return
		}
		logParts["Time"] = t
		return
	}
}

// DateParser simplifies the creation of GroupParser, when we just want to parse a date.
// See also DateParserToGroupParsers.
type DateParser func(date string) (t time.Time, err error)

// GroupParser parses a value and puts the result in logParts (the parser decides in which logParts key).
// Note that the value might be an empty string "" in some cases.
type GroupParser func(value string, logParts format.LogParts) error

// GroupParsers is a collection of GroupParser(s) indexed by group name
type GroupParsers map[string]GroupParser

func (f *RegexFormat) ParseLogLine(logLine []byte) (logParts format.LogParts, err error) {
	content := string(logLine)
	logParts = format.LogParts{
		"content": content,
	}

	// Apply regex
	regexpMatches := f.Regex.FindStringSubmatch(content)
	if len(regexpMatches) == 0 {
		err = fmt.Errorf("regexp didn't match for log: %v", content)
		return
	}

	// Collect captured groups into logParts
	for i, name := range f.Regex.SubexpNames() {
		if i == 0 {
			continue // Skip index 0. The index for the first sub-expression is 1.
		}
		// We avoid using f.Regex.SubexpIndex, which does a linear search
		logParts[name] = regexpMatches[i]
	}

	// Apply extra group parsers
	for groupName, groupParser := range f.GroupParsers {
		groupValue := logParts[groupName].(string)
		err = groupParser(groupValue, logParts)
		if err != nil {
			return
		}
	}

	return
}
