package formats

import (
	"fmt"
	"regexp"
	"strconv"
	"time"

	"gopkg.in/mcuadros/go-syslog.v2/format"
)

// StandardFormat for syslog (does this format have a specific name?)
// See for example:
// - https://www.elastic.co/blog/grokking-the-linux-authorization-logs
// - https://www.juniper.net/documentation/us/en/software/junos/network-mgmt/topics/topic-map/system-logging.html#id-interpreting-messages-generated-in-standard-format
var StandardFormat = FormatFromRegex(standardFormatRegex, buildStandardFormatGroupParsers())

// In some cases it comes with Priority, so it's optional
var standardFormatRegex = regexp.MustCompile(`^(?:<(?P<Priority>\d+)>)?(?P<Date>\w{3} +\d+ [^ ]+) +(?P<Host>[\w-.]+) +(?P<Process>[\w-]+)(?:\[(?P<ProcessId>\d+)])?: +(?P<Message>.+)`)

func buildStandardFormatGroupParsers() GroupParsers {
	parsers := DateParserToGroupParsers(ParseStandardFormatDate)
	parsers["Priority"] = parseStandardFormatPriority
	return parsers
}

func ParseStandardFormatDate(date string) (t time.Time, err error) {
	t, err = parseStandardDateWithGivenYear(date, time.Now().Year()) // guess it's current year
	if err == nil {
		if t.Month() > time.Now().Month() { // We might be in January 2022, receiving logs from last December 2021
			t, err = parseStandardDateWithGivenYear(date, time.Now().Year()-1) // must be year before
		}
	}
	return
}

func parseStandardDateWithGivenYear(date string, year int) (time.Time, error) {
	dateWithYear := fmt.Sprintf("%v %v", year, date)        // include year
	return time.Parse("2006 Jan _2 15:04:05", dateWithYear) // https://stackoverflow.com/q/25845172/1121497
}

func parseStandardFormatPriority(priority string, logParts format.LogParts) error {
	if priority != "" {
		priorityInt64, err := strconv.ParseInt(priority, 10, 64)
		if err != nil {
			return err
		}
		priorityInt := int(priorityInt64)
		logParts["Priority"] = priorityInt
		// Calculate severity and facility from priority, knowing that:
		// "The Priority value is calculated by first multiplying the Facility
		// number by 8 and then adding the numerical value of the Severity."
		// https://www.ietf.org/rfc/rfc3164.txt
		logParts["Severity"] = priorityInt % 8
		logParts["Facility"] = priorityInt / 8
	}
	return nil
}
