package parsers

import (
	"regexp"
	"strings"
	"time"

	"a.yandex-team.ru/security/osquery/osquery-sender/parser"
	"a.yandex-team.ru/security/osquery/osquery-sender/syslogparsing/formats"
	"a.yandex-team.ru/security/osquery/osquery-sender/syslogparsing/util"
)

// Huawei logs
// https://support.huawei.com/enterprise/en/doc/EDOC1000166533/97342dce/log-format-description

var Huawei = SyslogParser{
	Format:    formats.FormatFromRegex(huaweiRegex, formats.DateParserToGroupParsers(parseHuaweiDate)),
	FillEvent: fillHuaweiEvent,
}

var huaweiRegex = regexp.MustCompile(`(?P<Date>\w{3} +\d+ \d{4} [\d:+-]+) (?P<Host>[\w-.]+) %%(?P<Version>\d{2})(?P<Module>\w+)/(?P<Severity>\d+)/(?P<Brief>\w+)\((?P<InfoType>\w+)\):(?P<Values>[\w=-]+);(?P<Message>.+)`)

func fillHuaweiEvent(logParts *util.LogPartsWrapper, event *parser.ParsedEvent) {
	columns := map[string]interface{}{
		"time":      logParts.GetUnixTime("Time"),
		"hostname":  logParts.GetString("Host"),
		"version":   logParts.GetString("Version"),
		"module":    logParts.GetString("Module"),
		"brief":     logParts.GetString("Brief"),
		"info_type": logParts.GetString("InfoType"),
		"severity":  logParts.GetStringAsFloat64("Severity"),
	}
	event.Data[ColumnsDataKey] = columns

	// Example of values: CID=0x807a040a-alarmID=0x08520003-clearType=service_resume
	values := logParts.GetString("Values")
	util.SplitAndAddValues(values, "-", "=", columns, "id_values", "")

	// Example of message:
	// The interface status changes. (OperStatus=UP, Reason=Interface link is up, mainIfname=25GE1/0/41)
	parseMessage(logParts.GetString("Message"), columns)
}

func parseMessage(message string, data map[string]interface{}) {
	valuesIndex := strings.LastIndex(message, "(")
	if valuesIndex < 0 || message[len(message)-1:] != ")" {
		data["message"] = message
		return
	}
	data["message"] = message[0 : valuesIndex-1]
	values := message[valuesIndex+1 : len(message)-1]
	util.SplitAndAddValues(values, ", ", "=", data, "message_values", "")
}

func parseHuaweiDate(date string) (t time.Time, err error) {
	return time.Parse("Jan _2 2006 15:04:05-07:00", date) // // https://stackoverflow.com/q/25845172/1121497
}
