package parsers

import (
	"strings"

	"gopkg.in/mcuadros/go-syslog.v2"

	"a.yandex-team.ru/security/osquery/osquery-sender/parser"
	"a.yandex-team.ru/security/osquery/osquery-sender/syslogparsing/util"
)

// Juniper structured logs
// https://www.juniper.net/documentation/en_US/junos/topics/reference/general/syslog-interpreting-msg-generated-structured-data-format.html

var JuniperStructured = SyslogParser{
	Format:    syslog.RFC5424,
	FillEvent: fillJuniperStructuredEvent,
}

func fillJuniperStructuredEvent(logParts *util.LogPartsWrapper, event *parser.ParsedEvent) {
	columns := map[string]interface{}{
		"app_name": logParts.GetString("app_name"),
		"hostname": logParts.GetString("hostname"),
		"message":  logParts.GetString("message"),
		"time":     logParts.GetUnixTime("timestamp"),
		"proc_id":  logParts.GetString("proc_id"),
		"priority": logParts.GetIntAsFloat64("priority"),
		"version":  logParts.GetIntAsFloat64("version"),
		"facility": logParts.GetIntAsFloat64("facility"),
		"severity": logParts.GetIntAsFloat64("severity"),
	}
	event.Data[ColumnsDataKey] = columns

	// In the docs this part is called TAG
	tag := logParts.GetString("msg_id")
	if tag != "-" {
		columns["tag"] = tag
	}

	parseStructuredData(logParts.GetString("structured_data"), columns)
}

func parseStructuredData(structuredData string, data map[string]interface{}) {
	if structuredData == "-" {
		return // no data
	}
	// Example: [junos@2636.1.1.1.2.513 session-id="40" state="up" reason="Detect Timer Expiry."]
	if !strings.HasPrefix(structuredData, StructuredDataPrefix) || !strings.HasSuffix(structuredData, StructuredDataSuffix) {
		return // unexpected format
	}
	// Example: 1.1.1.2.513 session-id="40" state="up" reason="Detect Timer Expiry."
	content := structuredData[len(StructuredDataPrefix) : len(structuredData)-len(StructuredDataSuffix)]
	platform := content[0:strings.Index(content, " ")]
	data["platform"] = platform

	// Example: session-id="40" state="up" reason="Detect Timer Expiry."
	keyValuesStr := content[len(platform)+1:]
	keyValues := strings.FieldsFunc(keyValuesStr, makeQuoteAwareSplitter())
	for _, keyValue := range keyValues {
		// Don't use split, just in case the value contains '='
		equalSignIndex := strings.Index(keyValue, "=")
		key := keyValue[0:equalSignIndex]
		value := keyValue[equalSignIndex+2 : len(keyValue)-1]
		data[key] = value
	}
}

// Returns a splitter that splits at spaces but ignores spaces inside quotes (like in "hello world")
// https://scene-si.org/2017/09/02/parsing-strings-with-go/
func makeQuoteAwareSplitter() func(rune) bool {
	inQuote := false
	return func(c rune) bool {
		switch {
		case c == '"':
			inQuote = !inQuote
			return false
		case inQuote:
			return false
		default:
			return c == ' '
		}
	}
}

const StructuredDataPrefix = "[junos@2636."
const StructuredDataSuffix = "]"
