package parsers

import (
	"regexp"
	"strings"

	"a.yandex-team.ru/security/osquery/osquery-sender/parser"
	"a.yandex-team.ru/security/osquery/osquery-sender/syslogparsing/formats"
	"a.yandex-team.ru/security/osquery/osquery-sender/syslogparsing/util"
)

// Shell parser handles shell logs in standard format, focusing on sshd and sudo logs.
// It extracts more data from sshd login messages, and also from sudo commands.
// For other logs it just extracts the generic data.
var Shell = SyslogParser{
	Format:    formats.StandardFormat,
	FillEvent: fillShellEvent,
}

func fillShellEvent(logParts *util.LogPartsWrapper, event *parser.ParsedEvent) {
	message := logParts.GetString("Message")
	columns := map[string]interface{}{
		"time":     logParts.GetUnixTime("Time"),
		"hostname": logParts.GetString("Host"),
		"process":  logParts.GetString("Process"),
		"message":  message,
	}
	event.Data[ColumnsDataKey] = columns

	processID := logParts.GetString("ProcessId")
	if processID != "" {
		columns["proc_id"] = processID
	}

	switch columns["process"] {
	case "sshd":
		parseSshdMessage(message, columns)
	case "sudo":
		parseSudoMessage(message, columns)
	}

	// If Severity exists, it means that Priority was correctly parsed in StandardFormat
	if logParts.HasKey("Severity") {
		columns["priority"] = logParts.GetIntAsFloat64("Priority")
		columns["severity"] = logParts.GetIntAsFloat64("Severity")
		columns["facility"] = logParts.GetIntAsFloat64("Facility")
	}
}

var sudoCommandUserRegex = util.NewNamedRegex(regexp.MustCompile(`^(?P<User>[^ ]+) : (?P<Values>.*)`))

func parseSudoMessage(message string, data map[string]interface{}) {
	// Example messages:
	// ferran : TTY=pts/1 ; PWD=/home/ferran ; USER=root ; COMMAND=/bin/systemctl restart rsyslog.service
	// games : user NOT in sudoers ; TTY=pts/0 ; PWD=/home/ferran ; USER=root ; COMMAND=/bin/ls
	matches := sudoCommandUserRegex.FindStringSubmatch(message)
	if len(matches) <= 0 {
		return
	}
	delete(data, "message") // we will parse the message now
	username := matches[sudoCommandUserRegex.SubexpToIndex["User"]]
	values := matches[sudoCommandUserRegex.SubexpToIndex["Values"]]
	data["username"] = username
	util.SplitAndAddValues(values, " ; ", "=", data, "message", "sudo_")
}

func parseSshdMessage(message string, data map[string]interface{}) {
	if loginPrefixRegex.MatchString(message) {
		parseLoginMessage(message, data)
	} else if strings.HasPrefix(message, "Invalid user") {
		parseInvalidUserMessage(message, data)
	}
}

// Examples:
// Accepted publickey for vagrant from 192.168.33.1 port 58803 ssh2: RSA SHA256:sI+b5IW42U3aJpuaogCvoeFVV2bRmmR8f1e6HFe63Yg
var loginRegex = util.NewNamedRegex(regexp.MustCompile(`^(?P<Result>\w+) (?P<AuthType>\w+) for (?:invalid user )?(?P<User>.+?) from (?P<IP>[\w.:\[\]]+) port (?P<Port>\d+) ssh2(?:: )?(?P<Info>.*)$`))
var loginPrefixRegex = util.NewNamedRegex(regexp.MustCompile(`^(Accepted|Failed) (publickey|password)`))
var invalidUserRegex = util.NewNamedRegex(regexp.MustCompile(`^Invalid user (?P<User>.+?) from (?P<IP>[\w.:\[\]]+)(?P<Info>.*)$`))

func parseLoginMessage(message string, data map[string]interface{}) {
	matches := loginRegex.FindStringSubmatch(message)
	if len(matches) <= 0 {
		return
	}
	data["auth_result"] = matches[loginRegex.SubexpToIndex["Result"]]
	data["username"] = matches[loginRegex.SubexpToIndex["User"]]
	data["source_ip"] = matches[loginRegex.SubexpToIndex["IP"]]
	data["source_port"] = matches[loginRegex.SubexpToIndex["Port"]]
	data["auth_type"] = matches[loginRegex.SubexpToIndex["AuthType"]]
	info := matches[loginRegex.SubexpToIndex["Info"]]
	if data["auth_type"] == "publickey" {
		parsePublicKeyInfo(info, data)
	} else {
		if info != "" {
			data["message"] = info // unparsed part of the message
		} else {
			delete(data, "message") // we parsed all the message parts
		}
	}
}

func parseInvalidUserMessage(message string, data map[string]interface{}) {
	matches := invalidUserRegex.FindStringSubmatch(message)
	if len(matches) <= 0 {
		return
	}
	data["auth_result"] = "Failed"
	data["username"] = matches[invalidUserRegex.SubexpToIndex["User"]]
	data["source_ip"] = matches[invalidUserRegex.SubexpToIndex["IP"]]
	info := matches[invalidUserRegex.SubexpToIndex["Info"]]
	if info != "" {
		data["message"] = info // unparsed part of the message
	} else {
		delete(data, "message") // we parsed all the message parts
	}
}

func parsePublicKeyInfo(info string, data map[string]interface{}) {
	// info examples:
	// RSA-CERT ID type=session;requester=svartapetov; (serial 2869349027138243535) CA RSA SHA256:xQdCsb6SvRtPo9XvY5DYNhYGjq+J3PBE5HGRcwUZKv0
	// RSA SHA256:sI+b5IW42U3aJpuaogCvoeFVV2bRmmR8f1e6HFe63Yg
	indexAfterAuthType := strings.Index(info, " ")
	if indexAfterAuthType < 0 {
		data["message"] = info
		return
	}
	keyType := info[0:indexAfterAuthType]
	if keyType == "RSA" {
		data["auth_type"] = keyType
		data["rsa_fingerprint"] = info[indexAfterAuthType+1:]
		delete(data, "message") // we parsed all the message parts
	} else if keyType == "RSA-CERT" {
		data["auth_type"] = keyType
		parseRsaCertInfo(info[indexAfterAuthType+1:], data)
	} else {
		data["message"] = info
	}
}

var rsaCertInfoRegex = util.NewNamedRegex(regexp.MustCompile(`^ID (?P<ID>[^ ]+); \(serial (?P<Serial>\d+)\) (?P<Fingerprint>.*)$`))

func parseRsaCertInfo(info string, data map[string]interface{}) {
	// info example:
	// ID type=session;requester=svartapetov; (serial 2869349027138243535) CA RSA SHA256:xQdCsb6SvRtPo9XvY5DYNhYGjq+J3PBE5HGRcwUZKv0
	matches := rsaCertInfoRegex.FindStringSubmatch(info)
	if len(matches) <= 0 {
		return
	}
	data["cert_id"] = matches[rsaCertInfoRegex.SubexpToIndex["ID"]]
	data["cert_serial"] = matches[rsaCertInfoRegex.SubexpToIndex["Serial"]]
	data["rsa_fingerprint"] = matches[rsaCertInfoRegex.SubexpToIndex["Fingerprint"]]
	delete(data, "message") // we parsed all the message parts
}
