package syslogparsing

import (
	"fmt"
	"log"
	"reflect"
	"regexp"

	"gopkg.in/mcuadros/go-syslog.v2"
	"gopkg.in/mcuadros/go-syslog.v2/format"

	"a.yandex-team.ru/security/osquery/osquery-sender/config"
	"a.yandex-team.ru/security/osquery/osquery-sender/metrics"
	"a.yandex-team.ru/security/osquery/osquery-sender/parser"
	"a.yandex-team.ru/security/osquery/osquery-sender/sendmgr"
	"a.yandex-team.ru/security/osquery/osquery-sender/syslogparsing/formats"
	"a.yandex-team.ru/security/osquery/osquery-sender/syslogparsing/parsers"
	"a.yandex-team.ru/security/osquery/osquery-sender/syslogparsing/util"
)

const PseudoDstHostname = "syslog"

// StartSysLogServer a syslog server, parses events according to the conf, and sends each event to the EventHandler
func StartSysLogServer(serverConfig *config.SyslogServerConfig, eventHandler EventHandler) (server *syslog.Server, err error) {
	if serverConfig.Name == "" {
		return nil, fmt.Errorf("configuration doesn't have a name: %+v", serverConfig)
	}
	syslogParser, ok := syslogParsers[serverConfig.Parser]
	if !ok {
		return nil, fmt.Errorf("unknown parser in configuration: %+v", serverConfig)
	}

	log.Printf("Launching Syslog server with config: %+v\n", serverConfig)
	metrics.InitializeSyslogCounts(serverConfig.Name) // Add key (of count map) here, in a thread-safe way

	server = syslog.NewServer()
	server.SetFormat(syslogParser.Format)
	handler := &customHandler{
		serverConfig: serverConfig,
		syslogParser: syslogParser,
		eventHandler: eventHandler,
	}
	server.SetHandler(handler)
	if serverConfig.UseTCP {
		// TODO (ferran) 2021-12-17 - add EnableTLS option in SyslogServerConfig, and use existing TLS configs
		err = server.ListenTCP(serverConfig.Address)
	} else {
		err = server.ListenUDP(serverConfig.Address)
	}
	if err != nil {
		return
	}
	err = server.Boot()
	if err != nil {
		return
	}

	return
}

type EventHandler interface {
	Handle(*parser.ParsedEvent)
}

// Implements syslog.Handler
type customHandler struct {
	serverConfig *config.SyslogServerConfig
	syslogParser *parsers.SyslogParser
	eventHandler EventHandler
}

func (c customHandler) Handle(logParts format.LogParts, messageLength int64, err error) {
	metrics.IncrementSyslogMessageCount(c.serverConfig.Name)
	if err == nil {
		logPartsWrapper := &util.LogPartsWrapper{LogParts: logParts}
		event := prepareNewEvent(logPartsWrapper.GetClientHostname(), c.serverConfig.Name)
		c.syslogParser.FillEvent(logPartsWrapper, event)
		c.eventHandler.Handle(event)
	} else {
		metrics.IncrementSyslogErrorCount(c.serverConfig.Name)
		log.Printf("Syslog parse error: %v\n", err)
	}
}

const DefaultLogType = "result"
const DefaultAction = "added"

// Creates a parser.ParsedEvent with preset fields. Usually only Data should be set afterwards.
// https://st.yandex-team.ru/CLOUD-30054#61bc4ec503a13f1b4ffa7703
func prepareNewEvent(hostname string, name string) *parser.ParsedEvent {
	return &parser.ParsedEvent{
		Host:    hostname,
		LogType: DefaultLogType,
		Name:    name,
		Data: map[string]interface{}{
			// Match the osquery behavior
			"action": DefaultAction,
			"name":   name,
		},
	}
}

// Most logs don't conform to any official syslog format, so we create custom formats with regexp.
// See log formats here: https://st.yandex-team.ru/CLOUD-30054#61adff1dc654ba006667a6f9

var syslogParsers = map[string]*parsers.SyslogParser{
	"JuniperStructured": &parsers.JuniperStructured,
	"JuniperStandard":   &parsers.JuniperStandard,
	"Huawei":            &parsers.Huawei,
	"Cumulus":           &parsers.Cumulus,
	"Shell":             &parsers.Shell,
	"printLogLine":      {Format: NoFormat, FillEvent: printLogLine},  // for debugging
	"printLogParts":     {Format: NoFormat, FillEvent: printLogParts}, // for debugging
}

func printLogParts(logParts *util.LogPartsWrapper, event *parser.ParsedEvent) {
	log.Println("LogParts:")
	for k, v := range logParts.LogParts {
		log.Printf(" %v -> (%v) %v\n", k, reflect.TypeOf(v), v)
	}
}

func printLogLine(logParts *util.LogPartsWrapper, event *parser.ParsedEvent) {
	log.Println(logParts.GetString("content"))
}

// NoFormat just puts the whole log line in logParts["content"]
var NoFormat = formats.FormatFromRegex(regexp.MustCompile(`.*`), map[string]formats.GroupParser{})

// GetEventHandler returns a special EventHandler (for testing purposes)
func GetEventHandler(eventHandlerName string) EventHandler {
	switch eventHandlerName {
	case "PrintEvent":
		return &printEventHandler{}
	case "IgnoreEvent":
		return &ignoreEventHandler{}
	default:
		log.Fatalf("Unexpected event handler: %v", eventHandlerName)
		return nil
	}
}

type SendMgrHandler struct {
	Mgr *sendmgr.SendMgr
}

func (s *SendMgrHandler) Handle(event *parser.ParsedEvent) {
	s.Mgr.Log(PseudoDstHostname, []*parser.ParsedEvent{event})
}

type printEventHandler struct {
}

func (e printEventHandler) Handle(event *parser.ParsedEvent) {
	log.Printf("%v+\n", event)
}

type ignoreEventHandler struct {
}

func (e ignoreEventHandler) Handle(event *parser.ParsedEvent) {
	// nothing
}
