package util

import (
	"log"
	"regexp"
	"strconv"
	"strings"
	"time"

	"gopkg.in/mcuadros/go-syslog.v2/format"
)

// LogPartsWrapper adds convenience methods to prepare ParsedEvent.Data
// ParsedEvent.Data only accepts float64 or string, so we convert values to those types.
type LogPartsWrapper struct {
	LogParts format.LogParts
}

func (h *LogPartsWrapper) GetClientHostname() string {
	// The syslog server sets "client" = connection.RemoteAddr().String()
	client := h.LogParts["client"].(string)
	if i := strings.LastIndex(client, ":"); i > 1 {
		return client[:i]
	}
	return client
}

func (h *LogPartsWrapper) HasKey(key string) bool {
	_, ok := h.LogParts[key]
	return ok
}

func (h *LogPartsWrapper) GetIntAsFloat64(key string) float64 {
	return float64(h.LogParts[key].(int))
}

func (h *LogPartsWrapper) GetStringAsFloat64(key string) float64 {
	str := h.GetString(key)
	floatValue, err := strconv.ParseFloat(str, 64)
	if err != nil {
		log.Printf("Key %v contains value %v, which could not be parsed as float64: %v\n", key, str, err)
	}
	return floatValue
}

func (h *LogPartsWrapper) GetUnixTime(key string) float64 {
	return float64(h.LogParts[key].(time.Time).Unix())
}

func (h *LogPartsWrapper) GetString(key string) string {
	return h.LogParts[key].(string)
}

// SplitAndAddValues splits key/value pairs and adds them to data.
// For example, if values is like "key1=value1;key2=value2;key3=value3".
// then you can call splitAndAddValues(values, ";", "=", data) to add data["key1"] = "value1", etc.
// In case there are key-values that don't have the expected keyValueSep, they will be joined and set to data[nonKeyValuesKey].
func SplitAndAddValues(values string, valuesSep string, keyValueSep string, data map[string]interface{}, nonKeyValuesKey string, keyPrefix string) {
	var nonKeyValues []string
	for _, keyValue := range strings.Split(values, valuesSep) {
		sepIndex := strings.Index(keyValue, keyValueSep)
		if sepIndex < 1 || sepIndex == len(keyValue)-1 {
			nonKeyValues = append(nonKeyValues, keyValue)
			continue
		}
		key := keyValue[0:sepIndex]
		value := keyValue[sepIndex+1:]
		data[keyPrefix+key] = value
	}
	if len(nonKeyValues) > 0 {
		data[nonKeyValuesKey] = strings.Join(nonKeyValues, valuesSep)
	}
}

// NamedRegex adds SubexpToIndex, that is faster than using Regexp.SubexpIndex
type NamedRegex struct {
	*regexp.Regexp
	SubexpToIndex map[string]int
}

func NewNamedRegex(regex *regexp.Regexp) *NamedRegex {
	var nameToIndex = map[string]int{}
	for i, name := range regex.SubexpNames() {
		nameToIndex[name] = i
	}
	return &NamedRegex{
		Regexp:        regex,
		SubexpToIndex: nameToIndex,
	}
}
