#!/bin/sh

tag_file='/etc/osquery.tag'
flags_file='/etc/osquery/osquery.flags'
conf_file='/etc/osquery/osquery.conf'
paths_file='/etc/osquery/osquery.conf.d/40_file_paths.conf'
tag_decorator_file='/etc/osquery/osquery.conf.d/59_decorator_tag.conf'
packs_file='/etc/osquery/osquery.conf.d/60_packs.conf'
packs_path='/etc/osquery/packs'
sysd_dropin='/etc/osquery/limits/osquery-systemd-drop-in.service'
extensions_path='/usr/local/bin'
extensions_load='/etc/osquery/extensions.load'
extensions_sock='/var/osquery/osquery.em'

append_packs_after_line=2
append_paths_after_line=2
append_flags_after_line=1
append_shadow_paths_after_line=1

# cache variables for all heavy functions
_LIBSH_CACHE_METADATA_ENDPOINT=""


osquery_query() {
    /usr/bin/env osqueryi --disable_extensions=1 --disable_events=1 --header=0 --csv --header=0 --csv "$1"
}

get_osquery_tag() {
    local endpoint="$(metadata_endpoint)"
    if [ -n "${endpoint}" ]; then
        # --fail option to skip body of 404 error
        curl --fail -s --retry 3 -H "Metadata-Flavor: Google" "${endpoint}/computeMetadata/v1/instance/attributes/osquery_tag" || true
    fi
}

get_instance_id() {
    local endpoint="$(metadata_endpoint)"
    if [ -n "${endpoint}" ]; then
        # --fail option to skip body of 404 error
        curl -s --retry 3 -H "Metadata-Flavor: Google" "${endpoint}/computeMetadata/v1/instance/id" || true
    fi
}

get_environment() {
    local _env="$(test -f /etc/debian_chroot && (cat /etc/debian_chroot | tr a-z A-Z) || echo UKNWN)"
    if [ -n "${_env}" -a "x${_env}" != "xUKNWN" ]; then
        echo "${_env}"
        return 0
    fi

    local instance_id="${1:-$(get_instance_id)}"
    # local prefix="${instance_id:0:3}" - bashism
    local prefix="$(expr "x${instance_id}" : "x.\{,0\}\(.\{,3\}\)")"
    # env prefixes taken from https://bb.yandex-team.ru/projects/CLOUD/repos/cloud-go/browse/private-api/endpoint_ids
    case $prefix in
        fd8|btq|fhm|epd|ef3)
            _env="PROD"
            ;;
        fdv|b06|a7l|c8r|d9h)
            _env="PRE-PROD"
            ;;
        alk|bmq|fom)
            _env="ISRAEL"
            ;;
        c2p|dfs|fsa|all|bl8)
            _env="TESTING"
            ;;
        dcp|cn8|cbq)
            _env="DEV"
            ;;
        *)
            _env="UKNWN"
            ;;
    esac
    echo "${_env}"
}

add_pack () {
    sed -i "$append_packs_after_line a \         \"$1\":\"/etc/osquery/packs/$2.conf\"," $packs_file
}

del_pack_by_name () {
    # deleting pack with any preceding comma
    # 1. set label - :a
    # 2. push next line to buffer - N
    # 3. replace on buffer
    # 4. repeat if any unprocessed lines in file
    sed -i -e ":a" -e "N" -Ee "s/,?\n\s*\"$1\": \".*?\"//g" -e '$!ba' $packs_file
}

# TODO: taken from junk, should polish in next release
# https://a.yandex-team.ru/arc/trunk/arcadia/junk/axlodin/PKGBUILDS/osquery-ycloud-workstations-config/PKGBUILD?rev=7234593#L30
__set_flag() {
	local file="$1"
	local flag="$2" # actually flag pattern
	local value="$3"
	local position="${4:-1}" # after which flag insert new flag
	
	if grep -qEe "--${flag}[\=\s].*?" "${file}";
	then
		sed -i -Ee "s/--${flag}([\s\=]+).*/--${flag}\1${value}/g" "${file}"
	else
	    sed -i -e "/--${position}/a\--${flag}=${value}" "${file}"
	fi
}

set_flag() {
   	local file="${flags_file}"
	local flag="$1" # actually flag pattern
	local value="$2"
	local position="${3:-disable_extensions}" # after which flag insert new flag

    __set_flag "${file}" "${flag}" "${value}" "${position}"
}

false_flag () {
    set_flag "$1" "false"
}

enable_extension() {
    false_flag "disable_extensions"

    local ext_fullpath="${extensions_path}/$1.ext"
    if ! grep -Fxq "${ext_fullpath}" "${extensions_load}"; then
        echo "${extensions_path}/$1.ext" >> "${extensions_load}"
    fi
}

set_tls_hostname() {
    set_flag "tls_hostname" "$1"
}

set_splunk_tag() {
    sed -i "s/\"SELECT .* AS tag;\"/\"SELECT '$1' AS tag;\"/g" "${tag_decorator_file}"
}

# long to run, cache result if you can
__metadata_endpoint() {
    osquery_query "WITH meta(endpoint, ord)
                    AS (VALUES ('http://169.254.169.254', 1),
                               ('http://localhost:6770' , 2))
                   SELECT meta.endpoint
                     FROM ycloud_instance_metadata out
                     JOIN meta
                       ON meta.endpoint = out.metadata_endpoint
                    WHERE COALESCE(out.instance_id, '') <> ''
                    ORDER BY meta.ord LIMIT 1;
                  "
}

metadata_endpoint() {
    if [ -z "${_LIBSH_CACHE_METADATA_ENDPOINT}" ]; then
        _LIBSH_CACHE_METADATA_ENDPOINT=$(__metadata_endpoint)
    fi

    echo "${_LIBSH_CACHE_METADATA_ENDPOINT}"
}

osquery_version() {
    osquery_query "SELECT COALESCE((SELECT version FROM deb_packages WHERE name = 'osquery-vanilla' AND status = 'ii' LIMIT 1), (SELECT version FROM osquery_info)) AS version;"
}

osquery_table_exists() {
    local table_name="$1"
    
    osquery_query ".table ${table_name}" | \
        grep -q "${table_name}" && return 0 || return 1
}

_version_entry() {
    local pos="$1"
    local ver="$2"
    echo "${ver}" | cut -d'.' -f "${pos}"
}

version_major() {
    _version_entry 1 "${1}"
}

version_middle() {
    _version_entry 2 "${1}"
}
