package main

import (
	"fmt"
	"os"
	"path/filepath"
	"reflect"
	"runtime"
	"strings"

	"github.com/gofrs/uuid"
	"google.golang.org/protobuf/proto"

	porto "a.yandex-team.ru/infra/porto/api_go"
	portorpc "a.yandex-team.ru/infra/porto/proto"
)

var target = "/etc"

func genUUID() string {
	v4, err := uuid.DefaultGenerator.NewV4()
	if err != nil {
		panic(err)
	}

	return v4.String()
}

func fatalf(msg string, args ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, msg+"\n", args...)
	os.Exit(1)
}

type testCase func(portoApi porto.PortoAPI) error

func testLayerExport(portoApi porto.PortoAPI) error {
	// Create new volume to deal with place policy
	vol, err := portoApi.CreateVolume("", nil)
	if err != nil {
		return fmt.Errorf("can't create volume: %w", err)
	}

	random := genUUID()
	innerPath := filepath.Join(vol.GetPath(), random)
	outerPath := filepath.Join(strings.Replace(vol.GetPath(), "/porto/volume_", "/place/porto_volumes/", -1), "native", random)
	fmt.Printf("volume path: %s\n", vol.GetPath())
	fmt.Printf("inner place path: %s\n", innerPath)
	fmt.Printf("outer place path: %s\n", outerPath)

	if err := os.MkdirAll(innerPath, os.ModePerm); err != nil {
		return fmt.Errorf("can't create inner path dirs: %w", err)
	}

	// Export Layer
	if err := os.MkdirAll(filepath.Join(innerPath, "porto_layers"), os.ModePerm); err != nil {
		return fmt.Errorf("can't create porto_players dirs: %w", err)
	}

	if err := os.Symlink(target, filepath.Join(innerPath, "porto_layers", "test")); err != nil {
		return fmt.Errorf("can't create porto_layers/test dirs: %w", err)
	}

	cwd, err := os.Getwd()
	if err != nil {
		panic(err)
	}

	req := &portorpc.TPortoRequest{
		ExportLayer: &portorpc.TExportLayerRequest{
			Layer:   proto.String("test"),
			Volume:  proto.String(""),
			Place:   proto.String(outerPath),
			Tarball: proto.String(filepath.Join(cwd, "layer.tar.gz")),
		},
	}

	_, err = portoApi.Call(req)
	if err != nil {
		return fmt.Errorf("export fail: %w", err)
	}

	return nil
}

func testStorageExport(portoApi porto.PortoAPI) error {
	// Create new volume to deal with place policy
	vol, err := portoApi.CreateVolume("", nil)
	if err != nil {
		return fmt.Errorf("can't create volume: %w", err)
	}

	random := genUUID()
	innerPath := filepath.Join(vol.GetPath(), random)
	outerPath := filepath.Join(strings.Replace(vol.GetPath(), "/porto/volume_", "/place/porto_volumes/", -1), "native", random)
	fmt.Printf("volume path: %s\n", vol.GetPath())
	fmt.Printf("inner place path: %s\n", innerPath)
	fmt.Printf("outer place path: %s\n", outerPath)

	if err := os.MkdirAll(innerPath, os.ModePerm); err != nil {
		return fmt.Errorf("can't create inner path dirs: %w", err)
	}

	// Export Layer
	if err := os.MkdirAll(filepath.Join(innerPath, "porto_storage"), os.ModePerm); err != nil {
		return fmt.Errorf("can't create porto_storage dirs: %w", err)
	}

	if err := os.Symlink(target, filepath.Join(innerPath, "porto_storage", "test")); err != nil {
		return fmt.Errorf("can't create porto_storage/test dirs: %w", err)
	}

	cwd, err := os.Getwd()
	if err != nil {
		panic(err)
	}

	req := &portorpc.TPortoRequest{
		ExportStorage: &portorpc.TExportStorageRequest{
			Name:    proto.String("test"),
			Place:   proto.String(outerPath),
			Tarball: proto.String(filepath.Join(cwd, "storage.tar.gz")),
		},
	}

	_, err = portoApi.Call(req)
	if err != nil {
		return fmt.Errorf("export fail: %w", err)
	}

	return nil
}

func main() {
	if len(os.Args) > 1 {
		target = os.Args[1]
	}

	portoApi, err := porto.Dial()
	if err != nil {
		fatalf("porto connection fail: %v", err)
	}

	for _, tc := range []testCase{testLayerExport, testStorageExport} {
		fmt.Printf("\n\n\n----------  %s ----------\n", runtime.FuncForPC(reflect.ValueOf(tc).Pointer()).Name())
		if err := tc(portoApi); err != nil {
			fmt.Printf("fail: %v\n", err)
			continue
		}

		fmt.Println("passed")
	}
}
