package main

import (
	"flag"
	"fmt"
	"os"
	"sort"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/semver"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/pocs/yadi/yadi-os-lib/pkgmanager"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/config"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/kernel"
)

type (
	kernelOpts struct {
		version string
	}
	packagesOpts struct {
		codename, pkgName, pkgVersion string
	}
)

var (
	kopts kernelOpts
	popts packagesOpts
)

func main() {
	kernelSubcmd := flag.NewFlagSet("kernel", flag.ExitOnError)
	kernelSubcmd.StringVar(&kopts.version, "version", "", "kernel version")

	packagesSubcmd := flag.NewFlagSet("packages", flag.ExitOnError)
	packagesSubcmd.StringVar(&popts.codename, "distro", "", "distro codename, like trusty, xenial, ...")
	packagesSubcmd.StringVar(&popts.pkgName, "pkg", "", "name of package")
	packagesSubcmd.StringVar(&popts.pkgVersion, "version", "", "version of package")

	if len(os.Args) == 1 {
		kernelSubcmd.Usage()
		packagesSubcmd.Usage()
		return
	}

	analyzer, err := analyze.New(analyze.Options{
		FeedURI:         config.FeedURI,
		MinimumSeverity: 0,
	})
	if err != nil {
		simplelog.Error("can't create analyzer", "err", err.Error())
		return
	}

	switch os.Args[1] {
	case "kernel":
		if err := kernelSubcmd.Parse(os.Args[2:]); err != nil {
			fmt.Print(err.Error())
			os.Exit(2)
		}
		analyzeKernel(analyzer, kopts)
		os.Exit(0)
	case "packages":
		if err := packagesSubcmd.Parse(os.Args[2:]); err != nil {
			fmt.Print(err.Error())
			os.Exit(2)
		}
		analyzePackages(analyzer, popts)
		os.Exit(0)
	default:
		simplelog.Error("invalid command", "cmd", os.Args[1])
		os.Exit(2)
	}
}

func analyzePackages(analyzer *analyze.Analyzer, opts packagesOpts) {
	mngr := pkgmanager.New(opts.codename, opts.pkgName, opts.pkgVersion)
	packagesResult, err := analyzer.CheckPackages(mngr)
	if err != nil {
		simplelog.Error("failed to check packages", "err", err.Error())
		return
	}

	for _, pkg := range packagesResult {
		fmt.Println(pkg.Package)
		if len(pkg.Vulnerabilities) == 0 {
			fmt.Println("there are no vulnerabilities")
			continue
		}
		vulns := make(analyze.Issues, len(pkg.Vulnerabilities))
		copy(vulns, pkg.Vulnerabilities)
		sort.SliceStable(vulns, func(i, j int) bool {
			return vulns[i].CVSSScore > vulns[j].CVSSScore
		})
		for _, pkgVuln := range vulns {
			fmt.Printf("\t%s (%s), [%s], %s\n", pkgVuln.ID, pkgVuln.AffectedVersions, cvs.ToSeverity(pkgVuln.CVSSScore), pkgVuln.Reference)
		}
	}
}

func analyzeKernel(analyzer *analyze.Analyzer, opts kernelOpts) {
	linux, err := newKernel(opts.version)
	if err != nil {
		simplelog.Error("can't parse kernel version", "err", err.Error())
		return
	}

	kernelResult, err := analyzer.CheckKernel(*linux)
	if err != nil {
		simplelog.Error("check kernel error", err.Error())
	}
	if len(kernelResult) == 0 {
		return
	}
	vulns := make(analyze.Issues, len(kernelResult))
	copy(vulns, kernelResult)
	fmt.Printf("found %d issues for kernel (%s):\n", len(vulns), linux.RawRelease)

	sort.SliceStable(vulns, func(i, j int) bool {
		return vulns[i].CVSSScore > vulns[j].CVSSScore
	})
	for _, issue := range vulns {
		fmt.Printf("\t%s, [%s], %s\n", issue.ID, cvs.ToSeverity(issue.CVSSScore), issue.Reference)
	}
}

func newKernel(rawKernelRelease string) (*kernel.Kernel, error) {
	if idx := strings.Index(rawKernelRelease, "-"); idx > 0 {
		rawKernelRelease = rawKernelRelease[:idx]
	}

	kernelRelease, err := semver.NewVersion(rawKernelRelease)
	if err != nil {
		return nil, xerrors.Errorf("failed to parse kernel release: %w", err)
	}

	return &kernel.Kernel{
		RawRelease: rawKernelRelease,
		Release:    kernelRelease,
	}, nil
}
