import re
import os
import time
import logging
import startrek_client

from flask import Flask, request
from flask_restful import Api
from flask_restful import Resource

from jinja2 import Environment, FileSystemLoader


app = Flask(__name__)
api_var = Api()

ROOT_PATH = os.path.dirname(os.path.realpath(__file__))

log_path = "/var/log/secnotify/secnotify.log"
logging.basicConfig(filename=log_path,
                    level=logging.DEBUG,
                    format='%(asctime)s:%(module)s:%(name)s:%(levelname)s:%(message)s')
logging.debug("secnotify startup")
logger = logging.getLogger()
rotation_handler = logging.handlers.RotatingFileHandler(
                                                       filename=log_path,
                                                       maxBytes=33554432  # .5G
                                                       )
logger.addHandler(rotation_handler)

policy = {
            os.environ['impulse_secaudit_token']: {
                                                 "SECAUDIT": {"secaudit_scan_v2", },
                                                 "SECALERTS": {"ycloud_pcidss_dependencies_alert", },
                                                 "MARKETFRONTECH": {"generic_dependencies_alert", },
                                                 "CLOUD": {"ycloud_sast_report", },
                                                 "TAXIREL": {"taxirel_scan", },
                                                },
            os.environ['solomon_client_token']: {
                                                 "MARKETPCIDSS": {
                                                                  "market_pci_dss",
                                                                  "market_pci_dss_csp",
                                                                  "market_pci_dss_jsv"
                                                                }
                                              },
            os.environ['impulse_issue_creator_token']: {
                                                        "*": {
                                                            "impulse_results",
                                                        },
                                                       },
            os.environ['discovery_notification_token']: {
                                                        "VSSEC": {
                                                            "discovery_new_object",
                                                        },
                                                       }
         }

queue_blacklist = {
                    "SALARY",
                    "ROTATION"
                  }


class TrackerCommentHandle(Resource):
    def post(self):
        result = {
                    "message": "Server Error. Ask admin for details."
                }
        http_code = 500
        json_data = request.get_json(force=True)
        startrek_token = os.environ['startrek_token_basic']
        env = Environment(
                    loader=FileSystemLoader(
                        os.path.join(
                                        ROOT_PATH,
                                        'templates',
                                        'tracker'
                                    )
                                            )
                        )

        queue = json_data.get("issue_id").split('-')[0]
        if queue in queue_blacklist:
            result["message"] = "Not allowed to work with %s." % queue
            http_code = 401
            return result, http_code
        auth = request.headers.get('Authorization')
        if auth in policy:
            if (queue in policy[auth]) or ("*" in policy[auth]):
                templates = policy[auth].get(queue)
                if not templates:
                    templates = policy[auth].get("*", {})
                    startrek_token = os.environ['startrek_token_master']
                if json_data.get("template_id") in templates:
                    t = env.get_template(json_data["template_id"])
                    text = t.render(**json_data["template_parameters"])
                    startrek = startrek_client.Startrek(
                                 useragent='SecNotify by Service Security',
                                 base_url='https://st-api.yandex-team.ru',
                                 token=startrek_token
                                )
                    try:
                        issue = startrek.issues[json_data["issue_id"]]
                        issue.update(comment=text, ignore_version_change=True)
                    except startrek_client.exceptions.UnprocessableEntity as e:
                        return e.error_messages, 400
                    except startrek_client.exceptions.Forbidden as e:
                        return e.error_messages, 403
                    result["message"] = "Ok"
                    http_code = 200
                else:
                    result["message"] = "Wrong template id."
                    http_code = 416
            else:
                result["message"] = "Queue not found."
                http_code = 404
        else:
            result["message"] = "Unauthorized"
            http_code = 401

        return result, http_code


class TrackerIssueHandle(Resource):
    def post(self):
        result = {
                    "message": "Server Error. Ask admin for details."
                }
        http_code = 500
        json_data = request.get_json(force=True)
        startrek_token = os.environ['startrek_token_basic']
        env = Environment(
                    loader=FileSystemLoader(
                        os.path.join(
                                        ROOT_PATH,
                                        'templates',
                                        'tracker'
                                    )
                                            )
                        )
        if re.findall('\W', json_data["template_id"]):
            result["message"] = "Wrong template id."
            http_code = 416
            return result, http_code

        queue = json_data["issue_parameters"].get("queue")
        if queue in queue_blacklist:
            result["message"] = "Not allowed to work with %s." % queue
            http_code = 401
            return result, http_code

        auth = request.headers.get('Authorization')
        if auth in policy:
            if (queue in policy[auth]) or ("*" in policy[auth]):
                templates = policy[auth].get(queue)
                if not templates:
                    templates = policy[auth].get("*", {})
                    startrek_token = os.environ['startrek_token_master']
                if json_data.get("template_id") in templates:
                    t = env.get_template(json_data.get("template_id"))
                    description = t.render(
                                           **json_data.get(
                                                        "template_parameters"
                                                        )
                                           )
                    startrek = startrek_client.Startrek(
                                 useragent='SecNotify by Service Security',
                                 base_url='https://st-api.yandex-team.ru',
                                 token=startrek_token
                                )
                    try:
                        created_ticket = startrek.issues.create(
                                                           description=description,
                                                           type='task',
                                                           **json_data.get(
                                                                "issue_parameters",
                                                                {}
                                                                )
                                                          )
                    except startrek_client.exceptions.UnprocessableEntity as e:
                        return e.error_messages, 400
                    except startrek_client.exceptions.Forbidden as e:
                        return e.error_messages, 403
                    result["message"] = "Created %s" % created_ticket.key
                    result["issue_key"] = created_ticket.key
                    http_code = 200
                else:
                    result["message"] = "Wrong template id."
                    http_code = 416
            else:
                result["message"] = "Queue not found."
                http_code = 404
        else:
            result["message"] = "Unauthorized"
            http_code = 401

        return result, http_code


@app.after_request
def after_request(response):
    timestamp = time.strftime('[%Y-%b-%d %H:%M]')
    app.logger.error(
                     '%s %s %s %s %s %s %s %s',
                                               timestamp,
                                               request.remote_addr,
                                               request.method,
                                               request.scheme,
                                               request.full_path,
                                               request.data,
                                               response.status,
                                               response.data
                    )
    return response


api_var.add_resource(TrackerCommentHandle, '/tracker/comment')
api_var.add_resource(TrackerIssueHandle, '/tracker/issue')
api_var.init_app(app)

if __name__ == '__main__':
    app.logger.addHandler(logger)
    app.run(debug=False, port=3000)
