package commands

import (
	"encoding/base64"
	"errors"
	"fmt"
	"net/mail"
	"os"
	"os/exec"
	"path/filepath"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/sectools/internal/filespec"
	"a.yandex-team.ru/security/sectools/internal/fsutil"
	"a.yandex-team.ru/security/sectools/pkg/templates"
)

var npmCmd = &cobra.Command{
	Use: "npm",
}

var npmBuildArgs struct {
	Name      string
	Version   string
	Author    string
	Publish   bool
	AuthLogin string
}

var npmBuildCmd = &cobra.Command{
	Use: "build",
	RunE: func(_ *cobra.Command, args []string) error {
		if npmBuildArgs.Name == "" {
			return fmt.Errorf("--name is required")
		}

		if npmBuildArgs.Version == "" {
			return fmt.Errorf("--version is required")
		}

		toolName := npmBuildArgs.Name
		toolName = strings.TrimSuffix(toolName, "-bin")
		toolName = strings.TrimSuffix(toolName, "_bin")

		workDir := "work"
		_ = os.RemoveAll(workDir)
		if err := os.Mkdir(workDir, 0o755); err != nil {
			return fmt.Errorf("can't create work dir: %w", err)
		}

		buildPackage := func() error {
			tmpl, err := templates.NPM()
			if err != nil {
				return fmt.Errorf("can't get npm fs: %w", err)
			}

			var authorName, authorEmail string
			if npmBuildArgs.Author != "" {
				author, err := mail.ParseAddress(npmBuildArgs.Author)
				if err != nil {
					return fmt.Errorf("failed to parse author: %w", err)
				}
				authorName = author.Name
				authorEmail = author.Address
			}

			return templates.NewBuilder(tmpl,
				templates.WithPackageName(npmBuildArgs.Name),
				templates.WithToolName(toolName),
				templates.WithVersion(npmBuildArgs.Version),
				templates.WithAuthor(authorName, authorEmail),
			).Build(workDir)
		}

		publish := func() error {
			cwd, err := filepath.Abs(workDir)
			if err != nil {
				return fmt.Errorf("abs workdir: %w", err)
			}

			cmd := exec.Command("npm", "publish")
			cmd.Dir = cwd
			cmd.Stdout = os.Stdout
			cmd.Stderr = os.Stderr
			cmd.Env = append(os.Environ(),
				fmt.Sprintf("NPM_CONFIG_USERCONFIG=%s/.default_npmrc", cwd),
				fmt.Sprintf("NPM_LOGIN=%s", npmBuildArgs.AuthLogin),
				fmt.Sprintf("NPM_PASSWORD=%s", base64.StdEncoding.EncodeToString([]byte(os.Getenv("AUTH_PASSWORD")))),
			)

			return cmd.Run()
		}

		logger.Info("build package from template")
		if err := buildPackage(); err != nil {
			return fmt.Errorf("failed to build package: %w", err)
		}

		if len(args) == 0 {
			return errors.New("no files provided")
		}

		logger.Info("pack binaries")
		for _, inSpec := range args {
			logger.Info("pack binary", log.String("spec", inSpec))
			fileSpec, err := filespec.Parse(inSpec)
			if err != nil {
				return fmt.Errorf("can't parse file spec %q: %w", inSpec, err)
			}

			if fileSpec.Arch != "amd64" {
				// not supported yet
				logger.Info("ignore binary", log.String("spec", inSpec), log.String("reason", "unsupported arch"))
				continue
			}

			var targetName string
			switch fileSpec.Platform {
			case "linux":
				targetName = fmt.Sprintf("nix_%s", toolName)
			case "darwin":
				targetName = fmt.Sprintf("osx_%s", toolName)
			case "windows":
				targetName = fmt.Sprintf("win_%s.exe", toolName)
			default:
				return fmt.Errorf("unsupported platform: %s", fileSpec.Platform)
			}

			if err := fsutil.CopyFile(fileSpec.Filepath, filepath.Join(workDir, "vendor", targetName)); err != nil {
				return fmt.Errorf("can't copy binary %q: %w", inSpec, err)
			}
			logger.Info("binary packed")
		}

		if !npmBuildArgs.Publish {
			logger.Info("done")
			return nil
		}

		logger.Info("publish package")
		if err := publish(); err != nil {
			return fmt.Errorf("can't publish package: %w", err)
		}

		logger.Info("done")
		return nil
	},
}

func init() {
	npmCmd.AddCommand(npmBuildCmd)
	flags := npmBuildCmd.PersistentFlags()
	flags.StringVar(&npmBuildArgs.Name, "name", "", "package name")
	flags.StringVar(&npmBuildArgs.Version, "version", "", "package version")
	flags.StringVar(&npmBuildArgs.Author, "author", "", "author in RFC 5322 format")
	flags.BoolVar(&npmBuildArgs.Publish, "publish", false, "publish package (must provide --auth-login and env[AUTH_PASSWORD])")
	flags.StringVar(&npmBuildArgs.AuthLogin, "auth-login", os.Getenv("AUTH_LOGIN"), "login to authenticate in npm")
}
