package commands

import (
	"fmt"
	"io/ioutil"
	"net/mail"
	"os"
	"os/exec"
	"path/filepath"
	"strings"

	"github.com/mitchellh/go-homedir"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/sectools/internal/filespec"
	"a.yandex-team.ru/security/sectools/internal/fsutil"
	"a.yandex-team.ru/security/sectools/pkg/templates"
)

var pypiCmd = &cobra.Command{
	Use: "pypi",
}

var pypiBuildArgs struct {
	Name      string
	Version   string
	Author    string
	Publish   bool
	AuthLogin string
}

var pypiBuildCmd = &cobra.Command{
	Use: "build",
	RunE: func(_ *cobra.Command, args []string) error {
		if pypiBuildArgs.Name == "" {
			return fmt.Errorf("--name is required")
		}

		if pypiBuildArgs.Version == "" {
			return fmt.Errorf("--version is required")
		}

		toolName := pypiBuildArgs.Name
		toolName = strings.TrimSuffix(toolName, "-bin")
		toolName = strings.TrimSuffix(toolName, "_bin")

		workDir := "work"
		_ = os.RemoveAll(workDir)
		if err := os.Mkdir(workDir, 0o755); err != nil {
			return fmt.Errorf("can't create work dir: %w", err)
		}

		buildPackage := func() error {
			f, err := templates.PyPi()
			if err != nil {
				return fmt.Errorf("can't get pypi fs")
			}

			var authorName, authorEmail string
			if pypiBuildArgs.Author != "" {
				author, err := mail.ParseAddress(pypiBuildArgs.Author)
				if err != nil {
					return fmt.Errorf("failed to parse author: %w", err)
				}
				authorName = author.Name
				authorEmail = author.Address
			}

			return templates.NewBuilder(f,
				templates.WithPackageName(pypiBuildArgs.Name),
				templates.WithToolName(toolName),
				templates.WithVersion(pypiBuildArgs.Version),
				templates.WithAuthor(authorName, authorEmail),
			).Build(workDir)
		}

		writePypiRc := func() error {
			rcPath, err := homedir.Expand("~/.pypirc")
			if err != nil {
				return fmt.Errorf("failed to resolve .pypirc: %w", err)
			}

			data := `
[distutils]
index-servers =
    yandex

[yandex]
repository: https://pypi.yandex-team.ru/simple/
username: %s
password: %s
`
			data = fmt.Sprintf(data, pypiBuildArgs.AuthLogin, os.Getenv("AUTH_PASSWORD"))
			return ioutil.WriteFile(rcPath, []byte(data), 0o644)
		}

		removeRc := func() {
			rcPath, err := homedir.Expand("~/.pypirc")
			if err != nil {
				return
			}

			_ = os.RemoveAll(rcPath)
		}

		publish := func(platform string) error {
			cwd, err := filepath.Abs(workDir)
			if err != nil {
				return fmt.Errorf("abs workdir: %w", err)
			}

			env := os.Environ()

			upload := func() error {
				args := []string{"setup.py"}
				if platform == "src" {
					args = append(args, "sdist")
				} else {
					env = append(env, "TARGET_PLATFORM="+platform)
					args = append(args, "bdist_wheel", "--universal")
				}
				args = append(args, "upload", "-r", "yandex")
				cmd := exec.Command("python", args...)
				cmd.Stdout = os.Stdout
				cmd.Stderr = os.Stderr
				cmd.Env = env
				cmd.Dir = cwd

				return cmd.Run()
			}

			cleanup := func() error {
				cmd := exec.Command("python", "setup.py", "clean", "--all")
				cmd.Stdout = os.Stdout
				cmd.Stderr = os.Stderr
				cmd.Env = env
				cmd.Dir = cwd

				return cmd.Run()
			}

			if err := cleanup(); err != nil {
				return fmt.Errorf("cleanup fail: %w", err)
			}

			if err := upload(); err != nil {
				return fmt.Errorf("upload fail: %w", err)
			}

			return nil
		}

		logger.Info("build package from template")
		if err := buildPackage(); err != nil {
			return fmt.Errorf("failed to build package: %w", err)
		}

		logger.Info("pack binaries")
		platforms := []string{"src"}
		for _, inSpec := range args {
			logger.Info("pack binary", log.String("spec", inSpec))
			fileSpec, err := filespec.Parse(inSpec)
			if err != nil {
				return fmt.Errorf("can't parse file spec %q: %w", inSpec, err)
			}

			if fileSpec.Arch != "amd64" {
				// not supported yet
				logger.Info("ignore binary", log.String("spec", inSpec), log.String("reason", "unsupported arch"))
				continue
			}

			var targetName string
			switch fileSpec.Platform {
			case "linux":
				targetName = fmt.Sprintf("nix_%s", toolName)
			case "darwin":
				targetName = fmt.Sprintf("osx_%s", toolName)
			case "windows":
				targetName = fmt.Sprintf("win_%s.exe", toolName)
			default:
				return fmt.Errorf("unsupported platform: %s", fileSpec.Platform)
			}

			if err := fsutil.CopyFile(fileSpec.Filepath, filepath.Join(workDir, pypiBuildArgs.Name, "vendor", targetName)); err != nil {
				return fmt.Errorf("can't copy binary %q: %w", inSpec, err)
			}
			platforms = append(platforms, fileSpec.Platform)
			logger.Info("binary packed")
		}

		if !pypiBuildArgs.Publish {
			logger.Info("done")
			return nil
		}

		logger.Info("write ~/.pypirc")
		if err := writePypiRc(); err != nil {
			return fmt.Errorf("failed to write pipyrc: %w", err)
		}
		defer removeRc()

		logger.Info("publish package")
		for _, platform := range platforms {
			logger.Info("publish package", log.String("platform", platform))
			if err := publish(platform); err != nil {
				return fmt.Errorf("can't publish package: %w", err)
			}
		}

		logger.Info("done")
		return nil
	},
}

func init() {
	pypiCmd.AddCommand(pypiBuildCmd)
	flags := pypiBuildCmd.PersistentFlags()
	flags.StringVar(&pypiBuildArgs.Name, "name", "", "package name")
	flags.StringVar(&pypiBuildArgs.Version, "version", "", "package version")
	flags.StringVar(&pypiBuildArgs.Author, "author", "", "author in RFC 5322 format")
	flags.BoolVar(&pypiBuildArgs.Publish, "publish", false, "publish package (must provide --auth-login and env[AUTH_PASSWORD])")
	flags.StringVar(&pypiBuildArgs.AuthLogin, "auth-login", os.Getenv("AUTH_LOGIN"), "login to authenticate in pypi")
}
