package commands

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/sectools/internal/filespec"
	"a.yandex-team.ru/security/sectools/pkg/sectools"
)

var toolsUploadCmdArgs struct {
	Name      string
	Version   string
	ReleaseTo string
	Upstream  string
}

var toolsUploadCmd = &cobra.Command{
	Use:     "upload",
	PreRunE: setSecToolsToken,
	RunE: func(_ *cobra.Command, args []string) error {
		if toolsUploadCmdArgs.Version == "" {
			return errors.New("empty version")
		}

		svc, err := sectools.NewService(
			sectools.WithUpstream(toolsUploadCmdArgs.Upstream),
			sectools.WithAuthToken(toolsAuthToken),
		)
		if err != nil {
			return err
		}

		manifest := sectools.Manifest{
			Version:  toolsUploadCmdArgs.Version,
			Binaries: make(sectools.BinariesMap),
		}

		uploadFile := func(spec string) error {
			fileSpec, err := filespec.Parse(spec)
			if err != nil {
				return fmt.Errorf("parse file spec: %w", err)
			}

			tool := sectools.Tool{
				Name:     toolsUploadCmdArgs.Name,
				Version:  toolsUploadCmdArgs.Version,
				Platform: fileSpec.Platform,
				Arch:     fileSpec.Arch,
			}

			f, err := os.Open(fileSpec.Filepath)
			if err != nil {
				return err
			}
			defer func() { _ = f.Close() }()

			uploadedVersion, err := svc.UploadVersion(context.Background(), tool, f)
			if err != nil {
				return err
			}

			if _, ok := manifest.Binaries[fileSpec.Platform]; !ok {
				manifest.Binaries[fileSpec.Platform] = make(map[sectools.Arch]sectools.DownloadInfo)
			}

			manifest.Binaries[fileSpec.Platform][fileSpec.Arch] = uploadedVersion.DownloadInfo
			return nil
		}

		if len(args) == 0 {
			return errors.New("no files provided")
		}

		for _, fileSpec := range args {
			logger.Info("upload file", log.String("spec", fileSpec))
			err := uploadFile(fileSpec)
			if err != nil {
				return fmt.Errorf("can't upload tool file %q: %w", fileSpec, err)
			}
			logger.Info("file uploaded")
		}

		logger.Info("upload manifest", log.Any("manifest", manifest))
		uploadedManifest, err := svc.UploadManifest(context.Background(), toolsUploadCmdArgs.Name, manifest)
		if err != nil {
			return fmt.Errorf("can't upload manifest: %w", err)
		}
		logger.Info("manifest uploaded")

		if toolsUploadCmdArgs.ReleaseTo != "" {
			err := svc.Release(
				context.Background(),
				toolsUploadCmdArgs.Name,
				toolsUploadCmdArgs.Version,
				sectools.Channel(toolsUploadCmdArgs.ReleaseTo),
			)
			if err != nil {
				return err
			}
		}

		_ = json.NewEncoder(os.Stdout).Encode(uploadedManifest.DownloadInfo)
		return nil
	},
}

func init() {
	flags := toolsUploadCmd.PersistentFlags()
	flags.StringVar(&toolsUploadCmdArgs.Name, "name", "", "tool name")
	flags.StringVar(&toolsUploadCmdArgs.Version, "version", "", "tool version")
	flags.StringVar(&toolsUploadCmdArgs.Upstream, "upstream", sectools.DefaultUpstream, "sectools upstream")
	flags.StringVar(&toolsUploadCmdArgs.ReleaseTo, "release-to", "", "release tool to channel after upload")
}
