package commands

import (
	"context"
	"fmt"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/sectools/internal/server"
	"a.yandex-team.ru/security/sectools/internal/server/infra"
)

const shutdownDeadline = 30 * time.Second

var startCmd = &cobra.Command{
	Use: "start",
	RunE: func(_ *cobra.Command, _ []string) error {
		i, err := infra.New(cfg)
		if err != nil {
			return fmt.Errorf("failed to create infa: %w", err)
		}

		go func() {
			i.Start()
		}()

		srv, err := server.NewServer(i)
		if err != nil {
			return fmt.Errorf("failed create server: %w", err)
		}

		errChan := make(chan error, 1)
		okChan := make(chan struct{}, 1)
		go func() {
			if err := srv.ListenAndServe(cfg.Addr); err != nil {
				errChan <- fmt.Errorf("failed to start application: %w", err)
			} else {
				okChan <- struct{}{}
			}
		}()

		sigChan := make(chan os.Signal, 1)
		signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)

		defer i.Logger.Info("service stopped")

		shutdown := func(ctx context.Context) error {
			_ = i.Shutdown(ctx)
			return srv.Shutdown(ctx)
		}

		select {
		case <-sigChan:
			i.Logger.Info("shutting down gracefully by signal")

			ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
			defer cancel()

			return shutdown(ctx)
		case <-okChan:
			return nil
		case err := <-errChan:
			return err
		}
	},
}
