package config

import (
	"context"
	"errors"
	"os"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"
)

type Releaser struct {
	Name   string   `yaml:"sib"`
	Tools  []string `yaml:"tools"`
	Logins []string `yaml:"logins"`
}

type S3 struct {
	Endpoint        string `yaml:"endpoint"`
	Bucket          string `yaml:"bucket"`
	AccessKeyID     string `yaml:"access_key_id"`
	SecretAccessKey string `yaml:"secret_access_key"`
}

type Config struct {
	Addr         string     `yaml:"addr"`
	Dev          bool       `yaml:"dev"`
	S3           S3         `yaml:"s3"`
	PrivateTools []string   `yaml:"private_tools"`
	Releasers    []Releaser `yaml:"releasers"`
}

func LoadConfig(cfgPath string) (*Config, error) {
	loader := confita.NewLoader(
		env.NewBackend(),
		file.NewBackend(cfgPath),
	)

	cfg := Config{
		Addr: ":3000",
		S3: S3{
			SecretAccessKey: os.Getenv("S3_SECRET_ACCESS_KEY"),
		},
	}
	err := loader.Load(context.Background(), &cfg)
	if err != nil {
		return nil, err
	}

	if cfg.S3.SecretAccessKey == "" {
		return nil, errors.New("S3_SECRET_ACCESS_KEY must be provided")
	}

	return &cfg, err
}
