package api

import (
	"context"
	"fmt"
	"io"
	"net/http"
	"path"
	"strings"
	"sync"

	"github.com/blang/semver/v4"
	"github.com/klauspost/compress/zstd"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/sectools/internal/server/controller"
	"a.yandex-team.ru/security/sectools/internal/server/infra"
	"a.yandex-team.ru/security/sectools/internal/server/middlewares"
	"a.yandex-team.ru/security/sectools/internal/tools"
	"a.yandex-team.ru/security/sectools/pkg/sectools"
)

type API struct {
	*infra.Infra
	uriPrefix string
}

func NewController(infra *infra.Infra) (controller.Controller, error) {
	return &API{
		Infra: infra,
	}, nil
}

func (a *API) BuildRoute(prefix string, g *echo.Group) {
	authMiddlewares := []echo.MiddlewareFunc{
		middlewares.NewAuthMiddleware(a.Infra),
		a.checkToolAuth,
	}

	a.uriPrefix = prefix
	v1 := g.Group("/v1/release")
	v1.GET("/:tool/:platform/:version", a.v1Download)
	v1.GET("/:tool/:platform/:version/info", a.v1ReleaseInfo)
	v1.GET("/:tool/:platform/:version/check-latest", a.v1CheckLatest)
	v1.GET("/:tool/:platform/:version/:filename", a.v1Download)

	v2 := g.Group("/v2")
	v2.GET("/proxy/:tool/:version/:filename", a.storageProxy)
	v2.GET("/dumb-proxy/:tool/:version/:filename/:dumb", a.dumbStorageProxy)
	v2.PUT("/upload/tool/:tool/:version/:platform/:arch", a.uploadVersion, authMiddlewares...)
	v2.PUT("/upload/manifest/:tool/:version/"+sectools.ManifestFilename, a.uploadManifest, authMiddlewares...)
	v2.POST("/release/:tool", a.releaseTool, authMiddlewares...)
}

func (a *API) Shutdown(_ context.Context) {}

func (a *API) v1Download(c echo.Context) error {
	toolName := c.Param("tool")
	tool, ok := a.Tools.Tool(toolName)
	if !ok {
		return apiNotFound(c)
	}

	stable, ok := tool.Releases[sectools.ChannelStable]
	if !ok {
		return apiNotFound(c)
	}

	if stable.Version == c.QueryParam("current") {
		return c.String(http.StatusNotModified, "")
	}

	version := c.Param("version")
	if version == "latest" {
		release := tool.Releases[sectools.ChannelStable]
		if release.Version == "" {
			return apiNotFound(c)
		}

		version = release.Version
	}

	return a.dumpProxy(
		c,
		toolName,
		version,
		fmt.Sprintf("%s.zst", toolFileName(toolName, c.Param("platform"), "amd64")),
	)
}

func (a *API) v1ReleaseInfo(c echo.Context) error {
	toolName := c.Param("tool")
	tool, ok := a.Tools.Tool(toolName)
	if !ok {
		return apiNotFound(c)
	}

	version := c.Param("version")
	var release tools.Release
	if version == "latest" {
		release = tool.Releases[sectools.ChannelStable]
	} else {
		var err error
		release, err = a.Tools.ToolRelease(c.Request().Context(), toolName, version)
		if err != nil {
			return a.apiError(c, err)
		}
	}

	if release.Version == "" {
		return apiNotFound(c)
	}

	targetPlatform := c.Param("platform")
	var downloadURL string
	for _, p := range release.Platforms {
		if p.Platform == targetPlatform {
			downloadURL = p.DownloadInfo.DumbURL
			break
		}
	}

	if downloadURL == "" {
		return apiNotFound(c)
	}

	return c.JSON(http.StatusOK, echo.Map{
		"ok": true,
		"result": echo.Map{
			"platform":     targetPlatform,
			"version":      release.Version,
			"stable":       true,
			"download_url": downloadURL,
		},
		"error": nil,
	})
}

func (a *API) v1CheckLatest(c echo.Context) error {
	toolName := c.Param("tool")
	tool, ok := a.Tools.Tool(toolName)
	if !ok {
		return apiNotFound(c)
	}

	release, ok := tool.Releases[sectools.ChannelStable]
	if !ok {
		return apiNotFound(c)
	}

	currentVersion, err := semver.Parse(c.Param("version"))
	if err != nil {
		return a.apiError(c, fmt.Errorf("invalid version: %w", err))
	}

	targetVersion, err := semver.Parse(release.Version)
	if err != nil {
		a.Logger.Error("failed to parse latest tool version",
			log.String("url", c.Request().URL.String()),
			log.String("ver", release.Version),
			log.Error(err),
		)

		return c.JSON(http.StatusOK, echo.Map{
			"ok": true,
			"result": echo.Map{
				"is_latest": true,
				"latest":    "",
			},
			"error": nil,
		})
	}

	var isLatest bool
	if skipPatch := c.QueryParam("skip_patch"); skipPatch == "yes" {
		isLatest = currentVersion.Major == targetVersion.Major && currentVersion.Minor == targetVersion.Minor
	} else {
		isLatest = currentVersion.Compare(targetVersion) != -1
	}

	return c.JSON(http.StatusOK, echo.Map{
		"ok": true,
		"result": echo.Map{
			"is_latest": isLatest,
			"latest":    release.Version,
		},
		"error": nil,
	})
}

func (a *API) storageProxy(c echo.Context) error {
	filename := c.Param("filename")
	if strings.Contains(filename, "/") {
		return a.apiError(c, fmt.Errorf("invalid filename: %s", filename))
	}

	toolName := c.Param("tool")
	tool, ok := a.Tools.Tool(toolName)
	if !ok {
		return apiNotFound(c)
	}

	stable, ok := tool.Releases[sectools.ChannelStable]
	if !ok {
		return apiNotFound(c)
	}

	if stable.Version == c.QueryParam("current") {
		return c.String(http.StatusNotModified, "")
	}

	targetPath := path.Join(c.Param("tool"), c.Param("version"), filename)
	return a.Storage.Serve(c.Request().Context(), targetPath, c.Response())
}

func (a *API) dumbStorageProxy(c echo.Context) error {
	toolName := c.Param("tool")
	tool, ok := a.Tools.Tool(toolName)
	if !ok {
		return apiNotFound(c)
	}

	stable, ok := tool.Releases[sectools.ChannelStable]
	if !ok {
		return apiNotFound(c)
	}

	if stable.Version == c.QueryParam("current") {
		return c.String(http.StatusNotModified, "")
	}

	return a.dumpProxy(
		c,
		c.Param("tool"),
		c.Param("version"),
		c.Param("filename"),
	)
}

func (a *API) dumpProxy(c echo.Context, toolName, version, filename string) error {
	if strings.Contains(filename, "/") {
		return a.apiError(c, fmt.Errorf("invalid filename: %s", filename))
	}

	targetExt := path.Ext(filename)
	if strings.HasPrefix(targetExt, ".json") {
		c.Response().Header().Set(echo.HeaderContentType, "application/json")
	} else {
		c.Response().Header().Set(echo.HeaderContentType, "application/octet-stream")
	}

	var writer io.Writer
	switch {
	case strings.HasSuffix(targetExt, ".zst"):
		pipeR, pipeW := io.Pipe()
		rr, err := zstd.NewReader(pipeR)
		if err != nil {
			_ = pipeW.Close()
			_ = pipeR.Close()
			return a.apiError(c, err)
		}

		var wg sync.WaitGroup
		wg.Add(1)
		go func() {
			_, _ = io.Copy(c.Response(), rr)
			rr.Close()
			_ = pipeR.Close()
			wg.Done()
		}()

		defer func() {
			_ = pipeW.Close()
			wg.Wait()
		}()

		writer = pipeW
	default:
		writer = c.Response()
	}

	targetPath := path.Join(toolName, version, filename)
	return a.Storage.Download(c.Request().Context(), targetPath, writer)
}

func (a *API) uploadVersion(c echo.Context) error {
	req := struct {
		Tool     string
		Version  string
		Platform string
		Arch     string
	}{
		Tool:     c.Param("tool"),
		Version:  c.Param("version"),
		Platform: c.Param("platform"),
		Arch:     c.Param("arch"),
	}

	contentType := c.Request().Header.Get(echo.HeaderContentType)
	switch contentType {
	case "application/zstd":
	default:
		return a.apiError(c, fmt.Errorf("unexpected content type: %s", contentType))
	}

	if _, err := semver.Parse(req.Version); err != nil {
		return a.apiError(c, fmt.Errorf("invalid version %q: %w", req.Version, err))
	}

	exists, err := a.isManifestExists(c.Request().Context(), req.Tool, req.Version)
	if err != nil {
		return a.apiError(c, fmt.Errorf("can't check previous version: %w", err))
	}

	if exists {
		return a.apiError(c, fmt.Errorf("version %q already exists", req.Version))
	}

	fileName := toolFileName(req.Tool, req.Platform, req.Arch)
	targetPath := path.Join(req.Tool, req.Version, fileName+".zst")
	return a.upload(c, targetPath, fileName, contentType, c.Request().Body)
}

func (a *API) uploadManifest(c echo.Context) error {
	req := struct {
		Tool    string
		Version string
	}{
		Tool:    c.Param("tool"),
		Version: c.Param("version"),
	}

	if !isSemver(req.Version) {
		return a.apiError(c, fmt.Errorf("tool version must be in semver: %s", req.Version))
	}

	exists, err := a.isManifestExists(c.Request().Context(), req.Tool, req.Version)
	if err != nil {
		return a.apiError(c, fmt.Errorf("can't check previous version: %w", err))
	}

	if exists {
		return a.apiError(c, fmt.Errorf("version %q already deployed", req.Version))
	}

	targetPath := path.Join(req.Tool, req.Version, sectools.ManifestFilename)
	return a.upload(c, targetPath, sectools.ManifestFilename, "application/json", c.Request().Body)
}

func (a *API) upload(c echo.Context, targetPath, dumbFileName, contentType string, in io.Reader) error {
	storageURL, err := a.Storage.Upload(c.Request().Context(), targetPath, contentType, in)
	if err != nil {
		return a.apiError(c, err)
	}

	downloadURL := "https://" + c.Request().Host + path.Join(a.uriPrefix, "v2", "proxy", targetPath)
	dumbURL := "https://" + c.Request().Host + path.Join(a.uriPrefix, "v2", "dumb-proxy", targetPath, dumbFileName)
	return c.JSON(http.StatusOK, sectools.ServiceUploadRsp{
		URL:     downloadURL,
		FastURL: storageURL,
		DumbURL: dumbURL,
	})
}

func (a *API) releaseTool(c echo.Context) error {
	var req struct {
		FromVersion string `json:"from_version"`
		ToChannel   string `json:"to_channel"`
	}
	if err := c.Bind(&req); err != nil {
		return a.apiError(c, fmt.Errorf("can't parse request: %w", err))
	}

	fromVersion, err := semver.Parse(req.FromVersion)
	if err != nil {
		return a.apiError(c, fmt.Errorf("tool version must be in semver %q: %w", req.FromVersion, err))
	}

	tool := c.Param("tool")
	channel := sectools.Channel(req.ToChannel)
	switch channel {
	case sectools.ChannelStable, sectools.ChannelPrestable, sectools.ChannelTesting:
	default:
		return a.apiError(c, fmt.Errorf("unsupported channel: %s", req.ToChannel))
	}

	err = func() error {
		existedTool, ok := a.Tools.Tool(tool)
		if !ok {
			return nil
		}

		rel, ok := existedTool.Releases[channel]
		if !ok {
			return nil
		}

		currentVer, err := semver.Parse(rel.Version)
		if err != nil {
			return nil
		}

		if currentVer.Compare(fromVersion) == 1 {
			return a.apiError(c, fmt.Errorf("downgrading is prohibined: %s (current) > %s (target)", rel.Version, req.FromVersion))
		}

		return nil
	}()
	if err != nil {
		return a.apiError(c, err)
	}

	sourcePath := path.Join(tool, req.FromVersion, sectools.ManifestFilename)
	targetPath := path.Join(tool, req.ToChannel, sectools.ManifestFilename)
	return a.Storage.Copy(c.Request().Context(), sourcePath, targetPath)
}

func (a *API) isManifestExists(ctx context.Context, tool, version string) (bool, error) {
	targetPath := path.Join(tool, version, sectools.ManifestFilename)
	return a.Storage.Exists(ctx, targetPath)
}

func (a *API) apiError(c echo.Context, err error) error {
	a.Logger.Error("failed to process API request", log.String("url", c.Request().RequestURI), log.Error(err))

	return c.JSON(http.StatusInternalServerError, echo.Map{
		"result": nil,
		"error":  err.Error(),
	})
}

func (a *API) checkToolAuth(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		login, ok := c.Get(middlewares.UserLoginKey).(string)
		if !ok {
			return a.apiError(c, fmt.Errorf("unexpected user login: %v", c.Get(middlewares.UserLoginKey)))
		}

		toolName := c.Param("tool")
		if !a.Releasers.IsToolReleaser(toolName, login) {
			return a.apiError(c, fmt.Errorf("user %q is not a releaser for %q", login, toolName))
		}

		return next(c)
	}
}

func apiNotFound(c echo.Context) error {
	return c.JSON(http.StatusNotFound, echo.Map{
		"result": nil,
		"error":  "not found",
	})
}

func isSemver(version string) bool {
	_, err := semver.Parse(version)
	return err == nil
}

func toolFileName(name, platform, arch string) string {
	return fmt.Sprintf("%s-%s-%s", name, platform, arch)
}
