package infra

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/security/sectools/internal/config"
	"a.yandex-team.ru/security/sectools/internal/releasers"
	"a.yandex-team.ru/security/sectools/internal/storage"
	"a.yandex-team.ru/security/sectools/internal/tools"
)

type Infra struct {
	cfg       *config.Config
	TVM       *tvmtool.Client
	BlackBox  blackbox.Client
	Storage   *storage.Storage
	Releasers *releasers.Releasers
	Tools     *tools.Syncer
	Debug     bool
	Logger    log.Logger
}

func New(cfg *config.Config) (*Infra, error) {
	logLvl := log.InfoLevel
	if cfg.Dev {
		logLvl = log.DebugLevel
	}
	zlog, err := zap.NewDeployLogger(logLvl)
	if err != nil {
		panic(fmt.Sprintf("failed to create logger: %s", err))
	}

	out := &Infra{
		cfg:    cfg,
		Debug:  cfg.Dev,
		Logger: zlog,
	}

	return out, out.initialize()
}

func (c *Infra) initialize() error {
	var err error
	if c.cfg.Dev {
		c.BlackBox, err = httpbb.NewClient(
			httpbb.Environment{
				BlackboxHost: "http://bugllocdev-vla.vla.yp-c.yandex.net",
			},
			httpbb.WithLogger(c.Logger.Structured()),
		)
		if err != nil {
			return fmt.Errorf("failed to create blackbox client: %w", err)
		}
	} else {
		c.TVM, err = tvmtool.NewDeployClient()
		if err != nil {
			return fmt.Errorf("failed to create tvm client: %w", err)
		}

		c.BlackBox, err = httpbb.NewIntranet(
			httpbb.WithTVM(c.TVM),
			httpbb.WithLogger(c.Logger.Structured()),
		)
		if err != nil {
			return fmt.Errorf("failed to create blackbox client: %w", err)
		}
	}

	c.Releasers = releasers.NewReleasers(c.cfg.Releasers)

	c.Storage, err = storage.NewStorage(c.cfg.S3)
	if err != nil {
		return fmt.Errorf("failed to create tools storage: %w", err)
	}

	c.Tools, err = tools.NewSyncer(c.Logger, c.Storage, c.cfg.PrivateTools)
	if err != nil {
		return fmt.Errorf("failed to create tools syncer: %w", err)
	}

	return nil
}

func (c *Infra) Start() {
	c.Tools.Start()
}

func (c *Infra) Shutdown(_ context.Context) error {
	c.Tools.Shutdown()
	return nil
}
