package middlewares

import (
	"errors"
	"fmt"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/security/sectools/internal/server/infra"
)

const (
	UserLoginKey = "X-Request-User-Login"
	UserIDKey    = "X-Request-User-ID"
)

func NewAuthMiddleware(i *infra.Infra) echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(e echo.Context) error {
			token := e.Request().Header.Get("Authorization")
			if len(token) <= 6 || token[:6] != "OAuth " {
				return errors.New("invalid Authorization header")
			}

			rsp, err := i.BlackBox.OAuth(
				e.Request().Context(),
				blackbox.OAuthRequest{
					OAuthToken: token[6:],
					UserIP:     getIP(e),
				})

			if err != nil {
				return fmt.Errorf("failed to check OAuth token: %w", err)
			}

			e.Set(UserIDKey, rsp.User.ID)
			e.Set(UserLoginKey, rsp.User.Login)
			return next(e)
		}
	}
}

func getIP(e echo.Context) string {
	l7IP := e.Request().Header.Get("X-Forwarded-For-Y")
	if l7IP == "" {
		return e.RealIP()
	}
	return l7IP
}
