package server

import (
	"fmt"
	"html/template"
	"io"
	"path"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/sectools/assets"
	"a.yandex-team.ru/security/sectools/internal/tools"
)

type templateRenderer struct {
	templates *template.Template
}

func newTemplateRenderer() echo.Renderer {
	return &templateRenderer{
		templates: template.Must(
			template.New("").
				Funcs(template.FuncMap{
					"PlatformName": platformName,
					"DownloadURL":  downloadURL,
				}).
				ParseFS(assets.TemplatesFS(), "*.html"),
		),
	}
}

func (t *templateRenderer) Render(w io.Writer, name string, data interface{}, _ echo.Context) error {
	return t.templates.ExecuteTemplate(w, name, data)
}

func platformName(platform tools.Platform) string {
	switch platform.Platform {
	case "linux":
		return "GNU/Linux"
	case "darwin":
		return fmt.Sprintf("macOS (%s)", platform.Arch)
	case "windows":
		return "Windows"
	case "freebsd":
		return "FreeBSD"
	default:
		return platform.Platform
	}
}

func downloadURL(platform tools.Platform) string {
	filename := func() string {
		ext := path.Ext(platform.BinaryName)
		name := strings.TrimSuffix(platform.BinaryName, ext)
		return fmt.Sprintf("%s_%s_%s_%s%s", name, platform.Platform, platform.Arch, platform.Version, ext)
	}

	return platform.DownloadInfo.DumbURL + "/" + filename()
}
