package server

import (
	"context"
	"fmt"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/sectools/internal/server/controller"
	"a.yandex-team.ru/security/sectools/internal/server/controller/api"
	"a.yandex-team.ru/security/sectools/internal/server/controller/common"
	"a.yandex-team.ru/security/sectools/internal/server/controller/pages"
	"a.yandex-team.ru/security/sectools/internal/server/controller/static"
	"a.yandex-team.ru/security/sectools/internal/server/infra"
)

type (
	Server struct {
		infra       *infra.Infra
		echo        *echo.Echo
		controllers []controller.Controller
	}
)

func NewServer(infra *infra.Infra) (*Server, error) {
	out := &Server{
		infra: infra,
		echo:  echo.New(),
	}

	return out, out.initialize()
}

func (s *Server) initialize() error {
	s.echo.Debug = s.infra.Debug
	s.echo.HTTPErrorHandler = newHTTPErrorHandler(s.infra)
	s.echo.Renderer = newTemplateRenderer()

	controllers := map[string]func(i *infra.Infra) (controller.Controller, error){
		"":        pages.NewController,
		"/common": common.NewController,
		"/static": static.NewController,
		"/api":    api.NewController,
	}
	for prefix, f := range controllers {
		c, err := f(s.infra)
		if err != nil {
			return fmt.Errorf("failed to create contoller %q: %w", prefix, err)
		}

		c.BuildRoute(prefix, s.echo.Group(prefix))
		s.controllers = append(s.controllers, c)
	}

	return nil
}

func (s *Server) ListenAndServe(addr string) error {
	return s.echo.Start(addr)
}

func (s *Server) Shutdown(ctx context.Context) error {
	return s.echo.Shutdown(ctx)
}
