package storage

import (
	"context"
	"fmt"
	"io"
	"net/http"
	"net/http/httputil"
	"path"

	"cuelang.org/go/pkg/strings"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/security/sectools/internal/config"
)

type (
	Storage struct {
		cfg           config.S3
		s3Client      *s3.S3
		uploader      *s3manager.Uploader
		downloader    *s3manager.Downloader
		downloadProxy *httputil.ReverseProxy
	}
)

func NewStorage(cfg config.S3) (*Storage, error) {
	storage := &Storage{
		cfg: cfg,
	}

	if err := storage.initS3Client(); err != nil {
		return nil, err
	}

	if err := storage.initBucket(); err != nil {
		return nil, err
	}

	storage.initUploader()
	storage.initDownloader()
	storage.initDownloadProxy()

	return storage, nil
}

func (s *Storage) initDownloadProxy() {
	host := fmt.Sprintf("%s.s3.mds.yandex.net", s.cfg.Bucket)

	director := func(req *http.Request) {
		req.URL.Scheme = "https"
		req.URL.Host = host
	}

	s.downloadProxy = &httputil.ReverseProxy{
		Director: director,
	}
}

func (s *Storage) initBucket() error {
	if _, err := s.s3Client.HeadBucket(&s3.HeadBucketInput{Bucket: aws.String(s.cfg.Bucket)}); err != nil {
		_, err := s.s3Client.CreateBucket(&s3.CreateBucketInput{Bucket: aws.String(s.cfg.Bucket)})
		if err != nil {
			return err
		}
	}
	return nil
}

func (s *Storage) initS3Client() error {
	creds := credentials.NewStaticCredentials(
		s.cfg.AccessKeyID,
		s.cfg.SecretAccessKey,
		"",
	)
	_, err := creds.Get()
	if err != nil {
		return fmt.Errorf("bad credentials: %w", err)
	}

	cfg := aws.NewConfig().
		WithRegion("dummy").
		WithEndpoint(s.cfg.Endpoint).
		WithCredentials(creds)

	s3Session, err := session.NewSession()
	if err != nil {
		return fmt.Errorf("failed to create session: %w", err)
	}

	s.s3Client = s3.New(s3Session, cfg)
	return nil
}

func (s *Storage) initUploader() {
	s.uploader = s3manager.NewUploaderWithClient(s.s3Client, func(u *s3manager.Uploader) {
		u.PartSize = 32 * 1024 * 1024
	})
}

func (s *Storage) initDownloader() {
	s.downloader = s3manager.NewDownloaderWithClient(s.s3Client, func(d *s3manager.Downloader) {
		d.Concurrency = 1
		d.PartSize = 32 * 1024 * 1024
	})
}

func (s *Storage) Exists(ctx context.Context, path string) (bool, error) {
	_, err := s.s3Client.HeadObjectWithContext(ctx, &s3.HeadObjectInput{
		Bucket: aws.String(s.cfg.Bucket),
		Key:    aws.String(path),
	})
	if err != nil {
		if isAwsError(err, s3.ErrCodeNoSuchKey) || isAwsError(err, "NotFound") {
			return false, nil
		}

		return false, err
	}

	return true, nil
}

func (s *Storage) Upload(ctx context.Context, path, contentType string, data io.Reader) (string, error) {
	_, err := s.uploader.UploadWithContext(ctx, &s3manager.UploadInput{
		Bucket:      aws.String(s.cfg.Bucket),
		Key:         aws.String(path),
		ContentType: aws.String(contentType),
		Body:        data,
	})
	if err != nil {
		return "", err
	}

	return fmt.Sprintf("https://%s.s3.mds.yandex.net/%s", s.cfg.Bucket, path), nil
}

func (s *Storage) Serve(ctx context.Context, path string, response *echo.Response) error {
	uri := fmt.Sprintf("http://%s.s3.mds.yandex.net/%s", s.cfg.Bucket, path)
	req, err := http.NewRequestWithContext(ctx, "GET", uri, nil)
	if err != nil {
		return fmt.Errorf("failed to create proxy request: %w", err)
	}

	s.downloadProxy.ServeHTTP(response, req)
	return nil
}

func (s *Storage) Download(ctx context.Context, path string, out io.Writer) error {
	_, err := s.downloader.DownloadWithContext(ctx, FakeWriterAt{out}, &s3.GetObjectInput{
		Bucket: aws.String(s.cfg.Bucket),
		Key:    aws.String(path),
	})

	return err
}

func (s *Storage) List(ctx context.Context, path string) ([]string, error) {
	result, err := s.s3Client.ListObjectsWithContext(ctx, &s3.ListObjectsInput{
		Bucket:    aws.String(s.cfg.Bucket),
		Prefix:    ptr.String(path),
		Delimiter: ptr.String("/"),
	})
	if err != nil {
		return nil, err
	}

	out := make([]string, len(result.CommonPrefixes))
	for i, prefix := range result.CommonPrefixes {
		out[i] = strings.Trim(strings.TrimPrefix(*prefix.Prefix, path), "/")
	}

	return out, err
}

func (s *Storage) Copy(ctx context.Context, from, to string) error {
	_, err := s.s3Client.CopyObjectWithContext(ctx, &s3.CopyObjectInput{
		Bucket:     aws.String(s.cfg.Bucket),
		CopySource: aws.String(path.Join(s.cfg.Bucket, from)),
		Key:        aws.String(to),
	})
	return err
}

func isAwsError(err error, code string) bool {
	if err, ok := err.(awserr.Error); ok && err.Code() == code {
		return true
	}
	return false
}

type FakeWriterAt struct {
	w io.Writer
}

// WriteAt writes to the writer and ignores the offset
func (fw FakeWriterAt) WriteAt(p []byte, _ int64) (n int, err error) {
	return fw.w.Write(p)
}
